##############################################
# Replication Material 
# Article: "Strategic Postponement of Coalition Policymaking in European Parliamentary Democracies"
# Author: Xiao Lu
# European Journal of Political Research
# 2024
##############################################
rm(list=ls())
################
# load packages
library(circglmbayes)
library(dplyr)
library(coda)
library(ggplot2)
library(ggrepel)
library(ggthemes)
#######################
# LOAD MAIN DATASET
load("dat_raw.RData")
dat1 = subset(dat_raw,! dat_raw$country %in% c("Netherlands"))
# Calculate running time
start.time <- Sys.time()

## DESCRIPTIVE STATISTICS
# Figure 4
labels = dat1 %>% 
  group_by(country) %>%  
  summarise(xPos = density(month2ne)$x[which.max(density(month2ne)$y)],
            yPos = max((density(month2ne))$y))
ggplot(dat1, aes(x=month2ne,color= country,linetype = country)) + 
  geom_rect(
    aes(xmin = 12, xmax = 0),
    fill = "gray",
    col = "gray",
    ymin = -Inf, ymax = Inf, alpha = 0.8
  ) +
  geom_vline(xintercept = c(12),linetype = "dashed")+
  geom_density()+
  scale_x_reverse(breaks=c(0,12,24,36))+
  theme_bw()+
  theme(legend.position = "None")+
  xlab("Time Budget")+
  ylab("Kernel Density")+
  geom_label_repel(data=labels, aes(x=xPos, y=yPos, colour=country, label=country)) +
  geom_text(aes(x = 6, y = 0.005,label = "Last Year",color = "black"))
ggsave("Fig4.jpg",width = 8, height = 6)

# Figure 5
dat_tmp = dat1
dat_tmp$country = "All Countries Combined"
dat_tmp = rbind(dat1,dat_tmp)
dat_tmp$high_experienced_scrutiny = as.factor(dat_tmp$high_experienced_scrutiny)
pdf("Fig5.pdf", width = 8,height = 6)
ggplot(dat_tmp, aes(x=month2ne,group = high_experienced_scrutiny, fill=high_experienced_scrutiny)) + 
  geom_density(alpha = 0.7)+
  scale_x_reverse(breaks=c(0,12,24,36))+
  theme_bw()+
  theme(legend.position = "bottom")+
  xlab("Time Budget")+
  ylab("Kernel Density")+
  facet_wrap(vars(country))+
  scale_fill_grey(labels = c("Low Experienced Scrutiny", "High Experienced Scrutiny"))+
  guides(fill = guide_legend(title = ""))
dev.off()

####################
## MAIN ESTIMATES
# Model 1
set.seed(12345)
model_1 <- circGLM(bill_location ~ month2ne, data = dat1, Q = 10000, burnin = 1000)
print(model_1)
round(coef(model_1),3)
# Model 2
set.seed(12345)
model_2 <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne, data = dat1, Q = 10000, burnin = 1000)
print(model_2)
round(coef(model_2),3)
# Model 3
set.seed(12345)
model_3 <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne +
                     high_coalition_conflict + high_saliency + 
                     min_parl_perc +
                     oppos_conflict + gov_duration+
                     minority + median_oppos, data = dat1, Q = 10000, burnin = 1000, thin = 1)
print(model_3)
round(coef(model_3),3)
# Model 4
set.seed(12345)
model_4 <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne +
                     high_coalition_conflict*month2ne + high_saliency*month2ne + 
                     min_parl_perc +
                     oppos_conflict + gov_duration+
                     minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_4)
round(coef(model_4),3)
model4_chains = model_4$all_chains

###############################
# COUNTRY-SPECIFIC ANALYSIS
# Overall results
set.seed(12345)
model_month_0 <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne, data = dat1, Q = 10000, burnin = 1000)
print(model_month_0)
# Belgium
set.seed(12345)
model_month_bel <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne, 
                           data = subset(dat1,dat1$country == "Belgium"), Q = 10000, burnin = 1000)
print(model_month_bel)
# Czech Republic
set.seed(12345)
model_month_cze <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne 
                           + high_experienced_scrutiny*lastyear_in_term, 
                           data = subset(dat1,dat1$country == "Czech Republic"), Q = 10000, burnin = 1000)
print(model_month_cze)
# Denmark
set.seed(12345)
model_month_den <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne, 
                           data = subset(dat1,dat1$country == "Denmark"), Q = 10000, burnin = 1000)
print(model_month_den)
# Estonia
set.seed(12345)
model_month_est <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne, 
                           data = subset(dat1,dat1$country == "Estonia"), Q = 10000, burnin = 1000)
print(model_month_est)
# Finland
set.seed(12345)
model_month_fin <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne, 
                           data = subset(dat1,dat1$country == "Finland"), Q = 10000, burnin = 1000)
print(model_month_fin)
# Germany
set.seed(12345)
model_month_ger <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne, 
                           data = subset(dat1,dat1$country == "Germany"), Q = 10000, burnin = 1000)
print(model_month_ger)
# Hungary
set.seed(12345)
model_month_hun <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne, 
                           data = subset(dat1,dat1$country == "Hungary"), Q = 10000, burnin = 1000)
print(model_month_hun)
# Latvia
set.seed(12345)
model_month_lat <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne, 
                           data = subset(dat1,dat1$country == "Latvia"), Q = 10000, burnin = 1000)
print(model_month_lat)
# Norway
set.seed(12345)
model_month_nor <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne, 
                           data = subset(dat1,dat1$country == "Norway"), Q = 10000, burnin = 1000)
print(model_month_nor)
# Poland
set.seed(12345)
model_month_pol <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne, 
                           data = subset(dat1,dat1$country == "Poland"), Q = 10000, burnin = 1000)
print(model_month_pol)

## Plot effects
# Circular prediction function
mu_pred_fun = function(beta0,beta_d,x_d = NULL,beta_c,x_c){
  vpi = pi
  if (is.null(x_d)){
    mu = beta0 + 2*atan( x_c %*% beta_c )    
  } else{
    mu = beta0 +x_d %*% beta_d + 2*atan( x_c %*% beta_c )    
  }
  return(mu)
}
# All countries combined
tmp = mu_pred_fun(beta0 = 0, beta_c = t(model_month_0$all_chains[,4]),x_c = matrix(seq(48,0)), 
                  beta_d = t(model_month_0$all_chains[,5]), x_d = matrix(rep(1,49)))
tmp_mean = apply(tmp,1,mean)
tmp_q95 = apply(tmp,1,quantile, probs = c(0.025,0.975))
tmp_sim = cbind(tmp_mean,t(tmp_q95))
months_sim = as.data.frame(tmp_sim/2/pi*48)
names(months_sim) = c("mean","ql","qu")
months_sim$months = seq(48,0)
months_sim$country = "All countries combined"
# Belgium
tmp = mu_pred_fun(beta0 = 0, beta_c = t(model_month_bel$all_chains[,4]),x_c = matrix(seq(48,0)), 
                  beta_d = t(model_month_bel$all_chains[,5]), x_d = matrix(rep(1,49)))
tmp_mean = apply(tmp,1,mean)
tmp_q95 = apply(tmp,1,quantile, probs = c(0.025,0.975))
tmp_sim = cbind(tmp_mean,t(tmp_q95))
months_sim1 = as.data.frame(tmp_sim/2/pi*48)
names(months_sim1) = c("mean","ql","qu")
months_sim1$months = seq(48,0)
months_sim1$country = "Belgium"
# Czech Republic
tmp = mu_pred_fun(beta0 = 0, beta_c = t(model_month_cze$all_chains[,4]),x_c = matrix(seq(48,0)), 
                  beta_d = t(model_month_cze$all_chains[,5]), x_d = matrix(rep(1,49)))
tmp_mean = apply(tmp,1,mean)
tmp_q95 = apply(tmp,1,quantile, probs = c(0.025,0.975))
tmp_sim = cbind(tmp_mean,t(tmp_q95))
months_sim2 = as.data.frame(tmp_sim/2/pi*48)
names(months_sim2) = c("mean","ql","qu")
months_sim2$months = seq(48,0)
months_sim2$country = "Czech Republic"
# Denmark
tmp = mu_pred_fun(beta0 = 0, beta_c = t(model_month_den$all_chains[,4]),x_c = matrix(seq(48,0)), 
                  beta_d = t(model_month_den$all_chains[,5]), x_d = matrix(rep(1,49)))
tmp_mean = apply(tmp,1,mean)
tmp_q95 = apply(tmp,1,quantile, probs = c(0.025,0.975))
tmp_sim = cbind(tmp_mean,t(tmp_q95))
months_sim3 = as.data.frame(tmp_sim/2/pi*48)
names(months_sim3) = c("mean","ql","qu")
months_sim3$months = seq(48,0)
months_sim3$country = "Denmark"
# Estonia
tmp = mu_pred_fun(beta0 = 0, beta_c = t(model_month_est$all_chains[,4]),x_c = matrix(seq(48,0)), 
                  beta_d = t(model_month_est$all_chains[,5]), x_d = matrix(rep(1,49)))
tmp_mean = apply(tmp,1,mean)
tmp_q95 = apply(tmp,1,quantile, probs = c(0.025,0.975))
tmp_sim = cbind(tmp_mean,t(tmp_q95))
months_sim4 = as.data.frame(tmp_sim/2/pi*48)
names(months_sim4) = c("mean","ql","qu")
months_sim4$months = seq(48,0)
months_sim4$country = "Estonia"
# Finland
tmp = mu_pred_fun(beta0 = 0, beta_c = t(model_month_fin$all_chains[,4]),x_c = matrix(seq(48,0)), 
                  beta_d = t(model_month_fin$all_chains[,5]), x_d = matrix(rep(1,49)))
tmp_mean = apply(tmp,1,mean)
tmp_q95 = apply(tmp,1,quantile, probs = c(0.025,0.975))
tmp_sim = cbind(tmp_mean,t(tmp_q95))
months_sim5 = as.data.frame(tmp_sim/2/pi*48)
names(months_sim5) = c("mean","ql","qu")
months_sim5$months = seq(48,0)
months_sim5$country = "Finland"
# Germany
tmp = mu_pred_fun(beta0 = 0, beta_c = t(model_month_ger$all_chains[,4]),x_c = matrix(seq(48,0)), 
                  beta_d = t(model_month_ger$all_chains[,5]), x_d = matrix(rep(1,49)))
tmp_mean = apply(tmp,1,mean)
tmp_q95 = apply(tmp,1,quantile, probs = c(0.025,0.975))
tmp_sim = cbind(tmp_mean,t(tmp_q95))
months_sim6 = as.data.frame(tmp_sim/2/pi*48)
names(months_sim6) = c("mean","ql","qu")
months_sim6$months = seq(48,0)
months_sim6$country = "Germany"
# Hungary
tmp = mu_pred_fun(beta0 = 0, beta_c = t(model_month_hun$all_chains[,4]),x_c = matrix(seq(48,0)), 
                  beta_d = t(model_month_hun$all_chains[,5]), x_d = matrix(rep(1,49)))
tmp_mean = apply(tmp,1,mean)
tmp_q95 = apply(tmp,1,quantile, probs = c(0.025,0.975))
tmp_sim = cbind(tmp_mean,t(tmp_q95))
months_sim7 = as.data.frame(tmp_sim/2/pi*48)
names(months_sim7) = c("mean","ql","qu")
months_sim7$months = seq(48,0)
months_sim7$country = "Hungary"
# Latvia
tmp = mu_pred_fun(beta0 = 0, beta_c = t(model_month_lat$all_chains[,4]),x_c = matrix(seq(48,0)), 
                  beta_d = t(model_month_lat$all_chains[,5]), x_d = matrix(rep(1,49)))
tmp_mean = apply(tmp,1,mean)
tmp_q95 = apply(tmp,1,quantile, probs = c(0.025,0.975))
tmp_sim = cbind(tmp_mean,t(tmp_q95))
months_sim8 = as.data.frame(tmp_sim/2/pi*48)
names(months_sim8) = c("mean","ql","qu")
months_sim8$months = seq(48,0)
months_sim8$country = "Latvia"
# Norway
tmp = mu_pred_fun(beta0 = 0, beta_c = t(model_month_nor$all_chains[,4]),x_c = matrix(seq(48,0)), 
                  beta_d = t(model_month_nor$all_chains[,5]), x_d = matrix(rep(1,49)))
tmp_mean = apply(tmp,1,mean)
tmp_q95 = apply(tmp,1,quantile, probs = c(0.025,0.975))
tmp_sim = cbind(tmp_mean,t(tmp_q95))
months_sim10 = as.data.frame(tmp_sim/2/pi*48)
names(months_sim10) = c("mean","ql","qu")
months_sim10$months = seq(48,0)
months_sim10$country = "Norway"
# Poland
tmp = mu_pred_fun(beta0 = 0, beta_c = t(model_month_pol$all_chains[,4]),x_c = matrix(seq(48,0)), 
                  beta_d = t(model_month_pol$all_chains[,5]), x_d = matrix(rep(1,49)))
tmp_mean = apply(tmp,1,mean)
tmp_q95 = apply(tmp,1,quantile, probs = c(0.025,0.975))
tmp_sim = cbind(tmp_mean,t(tmp_q95))
months_sim11 = as.data.frame(tmp_sim/2/pi*48)
names(months_sim11) = c("mean","ql","qu")
months_sim11$months = seq(48,0)
months_sim11$country = "Poland"
# Combine all results
months_sim_all = rbind(
  months_sim1,
  months_sim2,
  months_sim3,
  months_sim4,
  months_sim5,
  months_sim6,
  months_sim7,
  months_sim8,
  months_sim10,
  months_sim11,
  months_sim
)
#####################
# FIGURE 6
library(ggplot2)
pdf("Fig6.pdf",width = 8,height = 7)
ggplot(months_sim_all,aes(x = months, y = mean))+
  geom_ribbon(aes(ymin = ql, ymax = qu), fill = "grey70") +
  geom_line()+
  geom_abline(slope = 0, intercept = 0, linetype = "dashed")+
  scale_x_reverse(breaks = c(0,12,24,36,48))+
  xlab("Time Budget")+
  ylab("Late Initiation in Month with High Experienced Scrutiny")+
  theme_bw()+
  facet_wrap(vars(country))
dev.off()

###############################
## SUBSTANTIVE EFFECTS
# Policy Divergence
tmp1 = mu_pred_fun(beta0 = 0, beta_c = t(model4_chains[,8]),x_c = matrix(seq(48,0)), 
                   beta_d = t(model4_chains[,10]), x_d = matrix(rep(1,49)))
tmp1_mean = apply(tmp1,1,mean)
tmp1_q95 = apply(tmp1,1,quantile, probs = c(0.025,0.975))
tmp1_sim = cbind(tmp1_mean,t(tmp1_q95))
months_sim1 = as.data.frame(tmp1_sim/2/pi*48)
names(months_sim1) = c("mean","ql","qu")
months_sim1$months = seq(48,0)
months_sim1$type = "High Coalition Policy Divergence"
# Saliency
tmp2 = mu_pred_fun(beta0 = 0, beta_c = t(model4_chains[,9]),x_c = matrix(seq(48,0)), 
                   beta_d = t(model4_chains[,11]), x_d = matrix(rep(1,49)))
tmp2_mean = apply(tmp2,1,mean)
tmp2_q95 = apply(tmp2,1,quantile, probs = c(0.025,0.975))
tmp2_sim = cbind(tmp2_mean,t(tmp2_q95))
months_sim2 = as.data.frame(tmp2_sim/2/pi*48)
names(months_sim2) = c("mean","ql","qu")
months_sim2$months = seq(48,0)
months_sim2$type = "High Saliency"
months_sim_sal_conf = rbind(months_sim1,months_sim2)
###############
# Figure 7
pdf("Fig7.pdf",width = 6,height = 4.5)
ggplot(months_sim_sal_conf,aes(x = months, y = mean))+
  geom_ribbon(aes(ymin = ql, ymax = qu), fill = "grey70") +
  geom_line()+
  geom_abline(slope = 0, intercept = 0, linetype = "dashed")+
  scale_x_reverse(breaks = c(0,12,24,36,48))+
  xlab("Time Budget")+
  ylab("Late Initiation in Month")+
  theme_bw()+
  facet_wrap(vars(type))
dev.off()

# Total running time
end.time = Sys.time()
time.taken = end.time - start.time
time.taken
# Time difference of 8.437833 mins
