############################################################################################
# Replication File for Simulations of the Game Theoretical Model
# Article: "Strategic Postponement of Coalition Policymaking in European Parliamentary Democracies"
# Author: Xiao Lu
# European Journal of Political Research
# 2024
############################################################################################
rm(list=ls())
# Calculate running time
start.time = Sys.time()

######################
# wlog. set K = 1
# t measured by days
# alpha discounted by years
# strategy 1:
dt1 = function(t){
  return(t/t)
}
# strategy 2:
dt2 = function(t){
  return(-2/Period * t + 2)
}
# strategy 3:
dt3 = function(t){
  return(2/Period * t)
}
# probability of scrutiny as a function of scrutiny cost
f_c = function(c){
  1/(1 + c^2)
}
# electoral benefit
f_alpha = function(t){
  alpha^(1 + 2*t/365)
}

#################################
# Simulation 1: alpha = 0.8
# Whole term
Period = 4* 365 # days
# Position taking benefits
B = 1
# minister loss from scrutiny
L = 0.5 * B
# loss from policy failure
R = 2 * B
# electoral benefits 
alpha = 0.8
# scrutiny payoff for coalition partner
D = 1
# scrutiny cost
C = 0.8*D
# scrutiny duration
T_s = 365/4
# simulated scrutiny duration
Tss = 0:250
# simulated time space
t = 1: Period
# payoffs
payoff_m1s = c()
payoff_m2s = c()
payoff_m3s = c()
payoff_c1s = c()
payoff_c2s = c()
payoff_c3s = c()
# simulations
for (i in 1:length(Tss)){
  T_s = Tss[i]
  t0 = t[which(Period - t < T_s)]
  t1 = t[which(t< Period - T_s)]
  #####################
  # Payoff of minister
  # policy gain
  policy_gain1 = sum(f_alpha(t1) * dt1(t1) * B)
  policy_gain2 = sum(f_alpha(t1) * dt2(t1) * B)
  policy_gain3 = sum(f_alpha(t1) * dt3(t1) * B)
  # scrutiny cost
  scrutiny_cost1 = sum(L * f_c(C) * dt1(t))
  scrutiny_cost2 = sum(L * f_c(C) * dt2(t))
  scrutiny_cost3 = sum(L * f_c(C) * dt3(t))
  # electoral benefits
  policy_elec1 = sum(R * f_c(C) * dt1(t0))
  policy_elec2 = sum(R * f_c(C) * dt2(t0))
  policy_elec3 = sum(R * f_c(C) * dt3(t0))
  # overall payoff
  payoff_m1 = policy_gain1 - scrutiny_cost1 + policy_elec1
  payoff_m2 = policy_gain2 - scrutiny_cost2 + policy_elec2
  payoff_m3 = policy_gain3 - scrutiny_cost3 + policy_elec3
  # Payoff of coalition partner
  # policy gains from scrutiny
  policy_gain_c1  = sum(f_alpha(t1) * D * f_c(C) * dt1(t1))
  policy_gain_c2  = sum(f_alpha(t1) * D * f_c(C) * dt2(t1))
  policy_gain_c3  = sum(f_alpha(t1) * D * f_c(C) * dt3(t1))
  # electoral benefits
  policy_fail_c1 = sum(R * f_c(C) * dt1(t0))
  policy_fail_c2 = sum(R * f_c(C) * dt2(t0))
  policy_fail_c3 = sum(R * f_c(C) * dt3(t0))
  # overall payoff
  payoff_c1 = policy_gain_c1 + policy_fail_c1
  payoff_c2 = policy_gain_c2 + policy_fail_c2
  payoff_c3 = policy_gain_c3 + policy_fail_c3
  # store payoffs
  payoff_m1s = c(payoff_m1s,payoff_m1)
  payoff_m2s = c(payoff_m2s,payoff_m2)
  payoff_m3s = c(payoff_m3s,payoff_m3)
  payoff_c1s = c(payoff_c1s,payoff_c1)
  payoff_c2s = c(payoff_c2s,payoff_c2)
  payoff_c3s = c(payoff_c3s,payoff_c3)
}

# Figure A9
pdf("game_simulation1.pdf",width = 8,height = 6)
plot(Tss,payoff_m1s,type = "l",ylim = c(-100,900), lty = "dotted", lwd = 2.5,col = "black",
     xlab = "Scrutiny Intensity (by Days)",
     ylab = "Payoff",yaxes = "n")
lines(Tss,payoff_m2s,lty = "dashed", lwd = 2.5,col = "black")
lines(Tss,payoff_m3s,lty = "solid", lwd = 2.5,col = "black")
lines(Tss,payoff_c1s,col = "gray", lty = "dotted", lwd = 2.5)
lines(Tss,payoff_c2s,col = "gray", lty = "dashed", lwd = 2.5)
lines(Tss,payoff_c3s,col = "gray", lty = "solid", lwd = 2.5)
abline(v = 88,lwd = 1, lty = "dashed")
abline(v = 160,lwd = 1, lty = "dashed")
text(170, 145, "Payoff of the Minister")
text(170, 700, "Payoff of the Coalition Partner")
text(40,-100, "Equilibrium Exists")
text(130,-100, "No Equilibrium")
text(210,-100, "Equilibrium Exists")
legend("topleft", legend = c("Constant Timinig", "Early Initiation","Postponing"), lty = c("dotted","dashed","solid"),
       title = "Timing Strategy",lwd = c(2,2,2))
dev.off()


#################################
# Simulation 2: alpha = 0.95
# Whole term
Period = 4* 365 # days
# Position taking benefits
B = 1
# minister loss from scrutiny
L = 0.5 * B
# loss from policy failure
R = 2 * B
# electoral benefits 
alpha = 0.95
# scrutiny payoff for coalition partner
D = 1
# scrutiny cost
C = 0.8*D
# scrutiny duration
T_s = 365/4
# simulated scrutiny duration
Tss = 0:250
# simulated time space
t = 1: Period
# payoffs
payoff_m1s = c()
payoff_m2s = c()
payoff_m3s = c()
payoff_c1s = c()
payoff_c2s = c()
payoff_c3s = c()
# simulations
for (i in 1:length(Tss)){
  T_s = Tss[i]
  t0 = t[which(Period - t < T_s)]
  t1 = t[which(t< Period - T_s)]
  #####################
  # Payoff of minister
  # policy gain
  policy_gain1 = sum(f_alpha(t1) * dt1(t1) * B)
  policy_gain2 = sum(f_alpha(t1) * dt2(t1) * B)
  policy_gain3 = sum(f_alpha(t1) * dt3(t1) * B)
  # scrutiny cost
  scrutiny_cost1 = sum(L * f_c(C) * dt1(t))
  scrutiny_cost2 = sum(L * f_c(C) * dt2(t))
  scrutiny_cost3 = sum(L * f_c(C) * dt3(t))
  # electoral benefits
  policy_elec1 = sum(R * f_c(C) * dt1(t0))
  policy_elec2 = sum(R * f_c(C) * dt2(t0))
  policy_elec3 = sum(R * f_c(C) * dt3(t0))
  # overall payoff
  payoff_m1 = policy_gain1 - scrutiny_cost1 + policy_elec1
  payoff_m2 = policy_gain2 - scrutiny_cost2 + policy_elec2
  payoff_m3 = policy_gain3 - scrutiny_cost3 + policy_elec3
  # Payoff of coalition partner
  # policy gains from scrutiny
  policy_gain_c1  = sum(f_alpha(t1) * D * f_c(C) * dt1(t1))
  policy_gain_c2  = sum(f_alpha(t1) * D * f_c(C) * dt2(t1))
  policy_gain_c3  = sum(f_alpha(t1) * D * f_c(C) * dt3(t1))
  # electoral benefits
  policy_fail_c1 = sum(R * f_c(C) * dt1(t0))
  policy_fail_c2 = sum(R * f_c(C) * dt2(t0))
  policy_fail_c3 = sum(R * f_c(C) * dt3(t0))
  # overall payoff
  payoff_c1 = policy_gain_c1 + policy_fail_c1
  payoff_c2 = policy_gain_c2 + policy_fail_c2
  payoff_c3 = policy_gain_c3 + policy_fail_c3
  # store payoffs
  payoff_m1s = c(payoff_m1s,payoff_m1)
  payoff_m2s = c(payoff_m2s,payoff_m2)
  payoff_m3s = c(payoff_m3s,payoff_m3)
  payoff_c1s = c(payoff_c1s,payoff_c1)
  payoff_c2s = c(payoff_c2s,payoff_c2)
  payoff_c3s = c(payoff_c3s,payoff_c3)
}

# Figure A10
pdf("game_simulation1.1.pdf",width = 8,height = 6)
plot(Tss,payoff_m1s,type = "l",ylim = c(500,1000), lty = "dotted", lwd = 2.5,col = "black",
     xlab = "Scrutiny Intensity (by Days)",
     ylab = "Payoff",yaxes = "n")
lines(Tss,payoff_m2s,lty = "dashed", lwd = 2.5,col = "black")
lines(Tss,payoff_m3s,lty = "solid", lwd = 2.5,col = "black")
lines(Tss,payoff_c1s,col = "gray", lty = "dotted", lwd = 2.5)
lines(Tss,payoff_c2s,col = "gray", lty = "dashed", lwd = 2.5)
lines(Tss,payoff_c3s,col = "gray", lty = "solid", lwd = 2.5)
abline(v = 58,lwd = 1, lty = "dashed")
abline(v = 150,lwd = 1, lty = "dashed")
text(50, 600, "Payoff of the Minister")
text(170, 900, "Payoff of the Coalition Partner")
text(30,500, "Equilibrium Exists")
text(100,500, "No Equilibrium")
text(210,500, "Equilibrium Exists")
legend("topleft", legend = c("Constant Timinig", "Early Initiation","Postponing"), lty = c("dotted","dashed","solid"),
       title = "Timing Strategy",lwd = c(2,2,2))
dev.off()

#################################
# Simulation 3: alpha = 0.7
# Whole term
Period = 4* 365 # days
# Position taking benefits
B = 1
# minister loss from scrutiny
L = 0.5 * B
# loss from policy failure
R = 2 * B
# electoral benefits 
alpha = 0.7
# scrutiny payoff for coalition partner
D = 1
# scrutiny cost
C = 0.8*D
# scrutiny duration
T_s = 365/4
# simulated scrutiny duration
Tss = 0:250
# simulated time space
t = 1: Period
# payoffs
payoff_m1s = c()
payoff_m2s = c()
payoff_m3s = c()
payoff_c1s = c()
payoff_c2s = c()
payoff_c3s = c()
# simulations
for (i in 1:length(Tss)){
  T_s = Tss[i]
  t0 = t[which(Period - t < T_s)]
  t1 = t[which(t< Period - T_s)]
  #####################
  # Payoff of minister
  # policy gain
  policy_gain1 = sum(f_alpha(t1) * dt1(t1) * B)
  policy_gain2 = sum(f_alpha(t1) * dt2(t1) * B)
  policy_gain3 = sum(f_alpha(t1) * dt3(t1) * B)
  # scrutiny cost
  scrutiny_cost1 = sum(L * f_c(C) * dt1(t))
  scrutiny_cost2 = sum(L * f_c(C) * dt2(t))
  scrutiny_cost3 = sum(L * f_c(C) * dt3(t))
  # electoral benefits
  policy_elec1 = sum(R * f_c(C) * dt1(t0))
  policy_elec2 = sum(R * f_c(C) * dt2(t0))
  policy_elec3 = sum(R * f_c(C) * dt3(t0))
  # overall payoff
  payoff_m1 = policy_gain1 - scrutiny_cost1 + policy_elec1
  payoff_m2 = policy_gain2 - scrutiny_cost2 + policy_elec2
  payoff_m3 = policy_gain3 - scrutiny_cost3 + policy_elec3
  # Payoff of coalition partner
  # policy gains from scrutiny
  policy_gain_c1  = sum(f_alpha(t1) * D * f_c(C) * dt1(t1))
  policy_gain_c2  = sum(f_alpha(t1) * D * f_c(C) * dt2(t1))
  policy_gain_c3  = sum(f_alpha(t1) * D * f_c(C) * dt3(t1))
  # electoral benefits
  policy_fail_c1 = sum(R * f_c(C) * dt1(t0))
  policy_fail_c2 = sum(R * f_c(C) * dt2(t0))
  policy_fail_c3 = sum(R * f_c(C) * dt3(t0))
  # overall payoff
  payoff_c1 = policy_gain_c1 + policy_fail_c1
  payoff_c2 = policy_gain_c2 + policy_fail_c2
  payoff_c3 = policy_gain_c3 + policy_fail_c3
  # store payoffs
  payoff_m1s = c(payoff_m1s,payoff_m1)
  payoff_m2s = c(payoff_m2s,payoff_m2)
  payoff_m3s = c(payoff_m3s,payoff_m3)
  payoff_c1s = c(payoff_c1s,payoff_c1)
  payoff_c2s = c(payoff_c2s,payoff_c2)
  payoff_c3s = c(payoff_c3s,payoff_c3)
}

# Figure A11
pdf("game_simulation1.2.pdf",width = 8,height = 6)
plot(Tss,payoff_m1s,type = "l",ylim = c(-300,700), lty = "dotted", lwd = 2.5,col = "black",
     xlab = "Scrutiny Intensity (by Days)",
     ylab = "Payoff",yaxes = "n")
lines(Tss,payoff_m2s,lty = "dashed", lwd = 2.5,col = "black")
lines(Tss,payoff_m3s,lty = "solid", lwd = 2.5,col = "black")
lines(Tss,payoff_c1s,col = "gray", lty = "dotted", lwd = 2.5)
lines(Tss,payoff_c2s,col = "gray", lty = "dashed", lwd = 2.5)
lines(Tss,payoff_c3s,col = "gray", lty = "solid", lwd = 2.5)
abline(v = 75,lwd = 1, lty = "dashed")
abline(v = 130,lwd = 1, lty = "dashed")
text(200, 00, "Payoff of the Minister")
text(170, 600, "Payoff of the Coalition Partner")
text(25,-260, "Equilibrium Exists")
text(100,-260, "No Equilibrium")
text(210,-260, "Equilibrium Exists")
legend("topleft", legend = c("Constant Timinig", "Early Initiation","Postponing"), lty = c("dotted","dashed","solid"),
       title = "Timing Strategy",lwd = c(2,2,2))
dev.off()


################################################
# Payoff illustrations with plots
# Figure A1
# Minister policy payoff illustration
p1 = function(x) 0.8^x
p2 = function(x) 0.8^x*(2 - 2*x/5)
p3 = function(x) 0.8^x*2*x/5
xs = seq(0,5,length.out = 100)
p1s = p1(xs)
p2s = p2(xs)
p3s = p3(xs)
pdf("minister_policy_payoff.pdf",width = 7,height = 5)
plot(NA, xlim = c(-0.5,5.5),ylim = c(0,2),xlab = "Government Term",ylab = "Payoff",xaxt = "n", yaxt = "n")
lines(xs,p1s,lwd = 2, lty = "dotted")
lines(xs,p2s,lwd = 2, lty = "dashed")
lines(xs,p3s,lwd = 2)
abline(v = 2.5, lty = "dashed")
abline(v = 5, lty = "dashed")
abline(v = 0, lty = "dashed")
abline(h = 1, lty = "dashed")
abline(h = 2, lty = "dashed")
abline(h = 0, lty = "dashed")
axis(side = 1, at = 2.5, labels = expression(T/2), tck = -0.01)
axis(side = 1, at = 5, labels = expression(T), tck = -0.01)
axis(side = 1, at = 0, labels = expression(0), tck = -0.01)
axis(side = 2, at = 1, labels = expression(K), tck = -0.01,las = 2)
axis(side = 2, at = 2, labels = expression(2*K), tck = -0.01,las = 2)
axis(side = 2, at = 0, labels = expression(0), tck = -0.01,las = 2)
text(4,0.8,"Postponing Strategy")
text(0.3,0.8,"Constant Timing")
text(1,1.8,"Early Initiation")
dev.off()

# Figure A2
# Minister Scrutiny cost illustration
p1 = function(x) 0.5*1/(1+0.8^2)*(x+1)^0
p2 = function(x) 0.5*1/(1+0.8^2)*(2 - 2*x/5)
p3 = function(x) 0.5*1/(1+0.8^2)*2*x/5
xs = seq(0,5,length.out = 100)
p1s = p1(xs)
p2s = p2(xs)
p3s = p3(xs)
pdf("minister_scrutiny_cost.pdf",width = 7,height = 5)
plot(NA, xlim = c(-0.5,5.5),ylim = c(0,1),xlab = "Government Term",ylab = "Cost",xaxt = "n", yaxt = "n")
lines(xs,p1s,lwd = 2, lty = "dotted")
lines(xs,p2s,lwd = 2, lty = "dashed")
lines(xs,p3s,lwd = 2)
abline(v = 2.5, lty = "dashed")
abline(v = 5, lty = "dashed")
abline(v = 0, lty = "dashed")
abline(h = 1, lty = "dashed")
abline(h = 0, lty = "dashed")
axis(side = 1, at = 2.5, labels = expression(T/2), tck = -0.01)
axis(side = 1, at = 5, labels = expression(T), tck = -0.01)
axis(side = 1, at = 0, labels = expression(0), tck = -0.01)
axis(side = 2, at = 1, labels = expression(K), tck = -0.01,las = 2)
axis(side = 2, at = 0, labels = expression(0), tck = -0.01,las = 2)
text(4,0.6,"Postponing Strategy")
text(0.8,0.35,"Constant Timing")
text(1,0.6,"Early Initiation")
dev.off()

# Figure A3
# Minister electoral benefit illustration
p1 = function(x) 2*1/(1+0.8^2)*(x+1)^0
p2 = function(x) 2*1/(1+0.8^2)*(2 - 2*x/5)
p3 = function(x) 2*1/(1+0.8^2)*2*x/5
xs = seq(0,5,length.out = 100)
p1s = p1(xs)
p2s = p2(xs)
p3s = p3(xs)
pdf("minister_electoral_benefit.pdf",width = 7,height = 5)
plot(NA, xlim = c(-0.5,5.5),ylim = c(0,2.5),xlab = "Government Term",ylab = "Payoff",xaxt = "n", yaxt = "n")
lines(xs,p1s,lwd = 2, lty = "dotted")
lines(xs,p2s,lwd = 2, lty = "dashed")
lines(xs,p3s,lwd = 2)
abline(v = 2.5, lty = "dashed")
abline(v = 5, lty = "dashed")
abline(v = 0, lty = "dashed")
abline(h = 1, lty = "dashed")
abline(h = 2, lty = "dashed")
abline(h = 0, lty = "dashed")
axis(side = 1, at = 2.5, labels = expression(T/2), tck = -0.01)
axis(side = 1, at = 5, labels = expression(T), tck = -0.01)
axis(side = 1, at = 0, labels = expression(0), tck = -0.01)
axis(side = 2, at = 1, labels = expression(K), tck = -0.01,las = 2)
axis(side = 2, at = 2, labels = expression(2*K), tck = -0.01,las = 2)
axis(side = 2, at = 0, labels = expression(0), tck = -0.01,las = 2)
text(4,2.4,"Postponing Strategy")
text(0.8,1.4,"Constant Timing")
text(1,2.4,"Early Initiation")
dev.off()

# Figure A4
# Overall payoff for the minister
p1 = function(x,t,ts) {
  if (t <= 5 - ts){
    0.8^x - 0.5*1/(1+0.8^2)*(x+1)^0
  } else {
    - 0.5*1/(1+0.8^2)*(x+1)^0 + 2*1/(1+0.8^2)*(x+1)^0
  }
}
p2 = function(x,t,ts){
  if (t <= 5 - ts){
    0.8^x*(2 - 2*x/5) - 0.5*1/(1+0.8^2)*(2 - 2*x/5)
  } else {
    - 0.5*1/(1+0.8^2)*(2 - 2*x/5) + 2*1/(1+0.8^2)*(2 - 2*x/5)
  }    
} 
p3 = function(x,t,ts) {
  if (t <= 5 - ts){
    0.8^x*2*x/5 - 0.5*1/(1+0.8^2)*2*x/5
  } else {
    - 0.5*1/(1+0.8^2)*2*x/5 + 2*1/(1+0.8^2)*2*x/5
  }   
} 
xs = seq(0,5,length.out = 100)
p1s = c()
p2s = c()
p3s = c()
for (i in 1:length(xs)){
  p1s[i] = p1(xs[i],xs[i],1)
  p2s[i] = p2(xs[i],xs[i],1)
  p3s[i] = p3(xs[i],xs[i],1)
}
pdf("minister_overall_payoff.pdf",width = 7,height = 5)
plot(NA, xlim = c(-0.5,5.5),ylim = c(0,2),xlab = "Government Term",ylab = "Payoff",xaxt = "n", yaxt = "n")
lines(xs,p1s,lwd = 2, lty = "dotted")
lines(xs,p2s,lwd = 2, lty = "dashed")
lines(xs,p3s,lwd = 2)
abline(v = 2.5, lty = "dashed")
abline(v = 5, lty = "dashed")
abline(v = 0, lty = "dashed")
abline(h = 1, lty = "dashed")
abline(h = 2, lty = "dashed")
abline(h = 0, lty = "dashed")
abline(v = 4, lty = "dashed",col = "red")
axis(side = 1, at = 2.5, labels = expression(T/2), tck = -0.01)
axis(side = 1, at = 5, labels = expression(T), tck = -0.01)
axis(side = 1, at = 0, labels = expression(0), tck = -0.01)
axis(side = 1, at = 4, labels = expression(T - T[s]), tck = -0.01)
axis(side = 2, at = 1, labels = expression(K), tck = -0.01,las = 2)
axis(side = 2, at = 2, labels = expression(2*K), tck = -0.01,las = 2)
axis(side = 2, at = 0, labels = expression(0), tck = -0.01,las = 2)
text(4,1.8,"Postponing Strategy")
text(0.8,0.4,"Constant Timing")
text(0.5,1.5,"Early Initiation")
dev.off()

# Figure 3
pdf("minister_cumulative_payoff.pdf",width = 7,height = 5)
plot(NA, xlim = c(-0.5,5.5),ylim = c(0,60),xlab = "Government Term",ylab = "Cumulative Payoff",xaxt = "n", yaxt = "n")
lines(xs,cumsum(p1s),lwd = 2, lty = "dotted")
lines(xs,cumsum(p2s),lwd = 2, lty = "dashed")
lines(xs,cumsum(p3s),lwd = 2)
abline(v = 2.5, lty = "dashed")
abline(v = 5, lty = "dashed")
abline(v = 0, lty = "dashed")
abline(v = 4, lty = "dashed",col = "red")
axis(side = 1, at = 2.5, labels = expression(T/2), tck = -0.01)
axis(side = 1, at = 5, labels = expression(T), tck = -0.01)
axis(side = 1, at = 0, labels = expression(0), tck = -0.01)
axis(side = 2, at = 0, labels = expression(0), tck = -0.01,las = 2)
axis(side = 1, at = 4, labels = expression(T - T[s]), tck = -0.01)
text(3.2,8,"Postponing Strategy")
text(3.2,23,"Constant Timing")
text(3.2,42,"Early Initiation")
dev.off()


################################################
# Coalition partner policy payoff illustration
# Figure A5
p1 = function(x) 0.8^x*1/(1+0.8^2)
p2 = function(x) 0.8^x*1/(1+0.8^2)*(2 - 2*x/5)
p3 = function(x) 0.8^x*1/(1+0.8^2)*2*x/5
xs = seq(0,5,length.out = 100)
p1s = p1(xs)
p2s = p2(xs)
p3s = p3(xs)
pdf("coalition_policy_payoff.pdf",width = 7,height = 5)
plot(NA, xlim = c(-0.5,5.5),ylim = c(0,1.5),xlab = "Government Term",ylab = "Payoff",xaxt = "n", yaxt = "n")
lines(xs,p1s,lwd = 2, lty = "dotted")
lines(xs,p2s,lwd = 2, lty = "dashed")
lines(xs,p3s,lwd = 2)
abline(v = 2.5, lty = "dashed")
abline(v = 5, lty = "dashed")
abline(v = 0, lty = "dashed")
abline(h = 1, lty = "dashed")
abline(h = 0, lty = "dashed")
axis(side = 1, at = 2.5, labels = expression(T/2), tck = -0.01)
axis(side = 1, at = 5, labels = expression(T), tck = -0.01)
axis(side = 1, at = 0, labels = expression(0), tck = -0.01)
axis(side = 2, at = 1, labels = expression(K), tck = -0.01,las = 2)
axis(side = 2, at = 0, labels = expression(0), tck = -0.01,las = 2)
text(4,0.5,"Postponing Strategy")
text(0.3,0.45,"Constant Timing")
text(1,1.1,"Early Initiation")
dev.off()

# Figure A6
# Coalition partner electoral benefit illustration
p1 = function(x) 2*1/(1+0.8^2)*(x+1)^0
p2 = function(x) 2*1/(1+0.8^2)*(2 - 2*x/5)
p3 = function(x) 2*1/(1+0.8^2)*2*x/5
xs = seq(0,5,length.out = 100)
p1s = p1(xs)
p2s = p2(xs)
p3s = p3(xs)
pdf("coalition_electoral_benefit.pdf",width = 7,height = 5)
plot(NA, xlim = c(-0.5,5.5),ylim = c(0,2.5),xlab = "Government Term",ylab = "Payoff",xaxt = "n", yaxt = "n")
lines(xs,p1s,lwd = 2, lty = "dotted")
lines(xs,p2s,lwd = 2, lty = "dashed")
lines(xs,p3s,lwd = 2)
abline(v = 2.5, lty = "dashed")
abline(v = 5, lty = "dashed")
abline(v = 0, lty = "dashed")
abline(h = 1, lty = "dashed")
abline(h = 2, lty = "dashed")
abline(h = 0, lty = "dashed")
axis(side = 1, at = 2.5, labels = expression(T/2), tck = -0.01)
axis(side = 1, at = 5, labels = expression(T), tck = -0.01)
axis(side = 1, at = 0, labels = expression(0), tck = -0.01)
axis(side = 2, at = 1, labels = expression(K), tck = -0.01,las = 2)
axis(side = 2, at = 2, labels = expression(2*K), tck = -0.01,las = 2)
axis(side = 2, at = 0, labels = expression(0), tck = -0.01,las = 2)
text(4,2.4,"Postponing Strategy")
text(0.8,1.4,"Constant Timing")
text(1,2.4,"Early Initiation")
dev.off()

# Figure A7
# Overall payoff for the coalition partner
p1 = function(x,t,ts){
  if (t <= 5 - ts){
    0.8^x*1/(1+0.8^2)
  } else {
    2*1/(1+0.8^2)*(x+1)^0
  }
} 
p2 = function(x,t,ts){
  if (t <= 5 - ts){
    0.8^x*1/(1+0.8^2)*(2 - 2*x/5)
  } else {
    2*1/(1+0.8^2)*(2 - 2*x/5)
  }
} 
p3 = function(x,t,ts){
  if (t <= 5 - ts){
    0.8^x*1/(1+0.8^2)*2*x/5
  } else {
    2*1/(1+0.8^2)*2*x/5
  }
} 
xs = seq(0,5,length.out = 100)
p1s = c()
p2s = c()
p3s = c()
for (i in 1:length(xs)){
  p1s[i] = p1(xs[i],xs[i],1)
  p2s[i] = p2(xs[i],xs[i],1)
  p3s[i] = p3(xs[i],xs[i],1)
}
pdf("coalition_overall_payoff.pdf",width = 7,height = 5)
plot(NA, xlim = c(-0.5,5.5),ylim = c(0,2.5),xlab = "Government Term",ylab = "Payoff",xaxt = "n", yaxt = "n")
lines(xs,p1s,lwd = 2, lty = "dotted")
lines(xs,p2s,lwd = 2, lty = "dashed")
lines(xs,p3s,lwd = 2)
abline(v = 2.5, lty = "dashed")
abline(v = 5, lty = "dashed")
abline(v = 0, lty = "dashed")
abline(h = 2, lty = "dashed")
abline(h = 1, lty = "dashed")
abline(h = 0, lty = "dashed")
abline(v = 4, lty = "dashed",col = "red")
axis(side = 1, at = 2.5, labels = expression(T/2), tck = -0.01)
axis(side = 1, at = 5, labels = expression(T), tck = -0.01)
axis(side = 1, at = 0, labels = expression(0), tck = -0.01)
axis(side = 2, at = 1, labels = expression(K), tck = -0.01,las = 2)
axis(side = 2, at = 2, labels = expression(2*K), tck = -0.01,las = 2)
axis(side = 2, at = 0, labels = expression(0), tck = -0.01,las = 2)
axis(side = 1, at = 4, labels = expression(T - T[s]), tck = -0.01)
text(4,2.4,"Postponing Strategy")
text(0.8,0.4,"Constant Timing")
text(1,1.1,"Early Initiation")
dev.off()

# Figure A8
pdf("coalition_cumulative_payoff.pdf",width = 7,height = 5)
plot(NA, xlim = c(-0.5,5.5),ylim = c(0,60),xlab = "Government Term",ylab = "Cumulative Payoff",xaxt = "n", yaxt = "n")
lines(xs,cumsum(p1s),lwd = 2, lty = "dotted")
lines(xs,cumsum(p2s),lwd = 2, lty = "dashed")
lines(xs,cumsum(p3s),lwd = 2)
abline(v = 2.5, lty = "dashed")
abline(v = 5, lty = "dashed")
abline(v = 0, lty = "dashed")
abline(v = 4, lty = "dashed",col = "red")
axis(side = 1, at = 2.5, labels = expression(T/2), tck = -0.01)
axis(side = 1, at = 5, labels = expression(T), tck = -0.01)
axis(side = 1, at = 0, labels = expression(0), tck = -0.01)
axis(side = 2, at = 0, labels = expression(0), tck = -0.01,las = 2)
axis(side = 1, at = 4, labels = expression(T - T[s]), tck = -0.01)
text(3.2,8,"Postponing Strategy")
text(3.2,23,"Constant Timing")
text(3.2,45,"Early Initiation")
dev.off()

#################################
# Simulation 1: alpha = 0.8
# High likelihood of bill passage in the next term
pass_lik = 0.9
# Reduced electoral benefit of the minister
discount_e = 1- pass_lik
# Whole term
Period = 4* 365 # days
# Position taking benefits
B = 1
# minister loss from scrutiny
L = 0.5 * B
# loss from policy failure
R = 2 * B
# electoral benefits 
alpha = 0.8
# scrutiny payoff for coalition partner
D = 1
# scrutiny cost
C = 0.8*D
# scrutiny duration
T_s = 365/4
# simulated scrutiny duration
Tss = 0:250
# simulated time space
t = 1: Period
# payoffs
payoff_m1s = c()
payoff_m2s = c()
payoff_m3s = c()
payoff_c1s = c()
payoff_c2s = c()
payoff_c3s = c()
# simulations
for (i in 1:length(Tss)){
  T_s = Tss[i]
  t0 = t[which(Period - t < T_s)]
  t1 = t[which(t< Period - T_s)]
  #####################
  # Payoff of minister
  # policy gain
  policy_gain1 = sum(f_alpha(t1) * dt1(t1) * B)
  policy_gain2 = sum(f_alpha(t1) * dt2(t1) * B)
  policy_gain3 = sum(f_alpha(t1) * dt3(t1) * B)
  # additional policy gain
  policy_gain1a = sum(f_alpha(t0) * dt1(t0) * B * pass_lik)
  policy_gain2a = sum(f_alpha(t0) * dt2(t0) * B * pass_lik)
  policy_gain3a = sum(f_alpha(t0) * dt3(t0) * B * pass_lik)
  # scrutiny cost
  scrutiny_cost1 = sum(L * f_c(C) * dt1(t))
  scrutiny_cost2 = sum(L * f_c(C) * dt2(t))
  scrutiny_cost3 = sum(L * f_c(C) * dt3(t))
  # electoral benefits
  policy_elec1 = sum(R * f_c(C) * dt1(t0) * discount_e)
  policy_elec2 = sum(R * f_c(C) * dt2(t0) * discount_e)
  policy_elec3 = sum(R * f_c(C) * dt3(t0) * discount_e)
  # overall payoff
  payoff_m1 = policy_gain1 - scrutiny_cost1 + policy_elec1 + policy_gain1a
  payoff_m2 = policy_gain2 - scrutiny_cost2 + policy_elec2 + policy_gain2a
  payoff_m3 = policy_gain3 - scrutiny_cost3 + policy_elec3 + policy_gain3a
  # Payoff of coalition partner
  # policy gains from scrutiny
  policy_gain_c1  = sum(f_alpha(t1) * D * f_c(C) * dt1(t1))
  policy_gain_c2  = sum(f_alpha(t1) * D * f_c(C) * dt2(t1))
  policy_gain_c3  = sum(f_alpha(t1) * D * f_c(C) * dt3(t1))
  # additional policy gains 
  policy_gain_c1a  = sum(f_alpha(t0) * D * f_c(C) * dt1(t0) * pass_lik)
  policy_gain_c2a  = sum(f_alpha(t0) * D * f_c(C) * dt2(t0) * pass_lik)
  policy_gain_c3a  = sum(f_alpha(t0) * D * f_c(C) * dt3(t0) * pass_lik)
  # electoral benefits
  policy_fail_c1 = sum(R * f_c(C) * dt1(t0) * discount_e)
  policy_fail_c2 = sum(R * f_c(C) * dt2(t0) * discount_e)
  policy_fail_c3 = sum(R * f_c(C) * dt3(t0) * discount_e)
  # overall payoff
  payoff_c1 = policy_gain_c1 + policy_fail_c1 + policy_gain_c1a
  payoff_c2 = policy_gain_c2 + policy_fail_c2 + policy_gain_c2a
  payoff_c3 = policy_gain_c3 + policy_fail_c3 + policy_gain_c3a
  # store payoffs
  payoff_m1s = c(payoff_m1s,payoff_m1)
  payoff_m2s = c(payoff_m2s,payoff_m2)
  payoff_m3s = c(payoff_m3s,payoff_m3)
  payoff_c1s = c(payoff_c1s,payoff_c1)
  payoff_c2s = c(payoff_c2s,payoff_c2)
  payoff_c3s = c(payoff_c3s,payoff_c3)
}

# Figure A14
pdf("game_simulation1_a.pdf",width = 8,height = 6)
plot(Tss,payoff_m1s,type = "l",ylim = c(-100,600), lty = "dotted", lwd = 2.5,col = "black",
     xlab = "Scrutiny Intensity (by Days)",
     ylab = "Payoff",yaxes = "n")
lines(Tss,payoff_m2s,lty = "dashed", lwd = 2.5,col = "black")
lines(Tss,payoff_m3s,lty = "solid", lwd = 2.5,col = "black")
lines(Tss,payoff_c1s,col = "gray", lty = "dotted", lwd = 2.5)
lines(Tss,payoff_c2s,col = "gray", lty = "dashed", lwd = 2.5)
lines(Tss,payoff_c3s,col = "gray", lty = "solid", lwd = 2.5)
text(170, 145, "Payoff of the Minister")
text(170, 400, "Payoff of the Coalition Partner")
text(130,-100, "Equilibrium Exists")
legend("topleft", legend = c("Constant Timinig", "Early Initiation","Postponing"), lty = c("dotted","dashed","solid"),
       title = "Timing Strategy",lwd = c(2,2,2))
dev.off()


#################################
# Maximum 500 days scrutiny
# half passage rate in the next term
# Simulation 1: alpha = 0.8
# Half likelihood of bill passage in the next term
pass_lik = 0.5
# Reduced electoral benefit of the minister
discount_e = 1- pass_lik
# Whole term
Period = 4* 365 # days
# Position taking benefits
B = 1
# minister loss from scrutiny
L = 0.5 * B
# loss from policy failure
R = 2 * B
# electoral benefits 
alpha = 0.8
# scrutiny payoff for coalition partner
D = 1
# scrutiny cost
C = 0.8*D
# scrutiny duration
T_s = 365/4
# simulated scrutiny duration
Tss = 0:500
# simulated time space
t = 1: Period
# payoffs
payoff_m1s = c()
payoff_m2s = c()
payoff_m3s = c()
payoff_c1s = c()
payoff_c2s = c()
payoff_c3s = c()
# simulations
for (i in 1:length(Tss)){
  T_s = Tss[i]
  t0 = t[which(Period - t < T_s)]
  t1 = t[which(t< Period - T_s)]
  #####################
  # Payoff of minister
  # policy gain
  policy_gain1 = sum(f_alpha(t1) * dt1(t1) * B)
  policy_gain2 = sum(f_alpha(t1) * dt2(t1) * B)
  policy_gain3 = sum(f_alpha(t1) * dt3(t1) * B)
  # additional policy gain
  policy_gain1a = sum(f_alpha(t0) * dt1(t0) * B * pass_lik)
  policy_gain2a = sum(f_alpha(t0) * dt2(t0) * B * pass_lik)
  policy_gain3a = sum(f_alpha(t0) * dt3(t0) * B * pass_lik)
  # scrutiny cost
  scrutiny_cost1 = sum(L * f_c(C) * dt1(t))
  scrutiny_cost2 = sum(L * f_c(C) * dt2(t))
  scrutiny_cost3 = sum(L * f_c(C) * dt3(t))
  # electoral benefits
  policy_elec1 = sum(R * f_c(C) * dt1(t0) * discount_e)
  policy_elec2 = sum(R * f_c(C) * dt2(t0) * discount_e)
  policy_elec3 = sum(R * f_c(C) * dt3(t0) * discount_e)
  # overall payoff
  payoff_m1 = policy_gain1 - scrutiny_cost1 + policy_elec1 + policy_gain1a
  payoff_m2 = policy_gain2 - scrutiny_cost2 + policy_elec2 + policy_gain2a
  payoff_m3 = policy_gain3 - scrutiny_cost3 + policy_elec3 + policy_gain3a
  # Payoff of coalition partner
  # policy gains from scrutiny
  policy_gain_c1  = sum(f_alpha(t1) * D * f_c(C) * dt1(t1))
  policy_gain_c2  = sum(f_alpha(t1) * D * f_c(C) * dt2(t1))
  policy_gain_c3  = sum(f_alpha(t1) * D * f_c(C) * dt3(t1))
  # additional policy gains 
  policy_gain_c1a  = sum(f_alpha(t0) * D * f_c(C) * dt1(t0) * pass_lik)
  policy_gain_c2a  = sum(f_alpha(t0) * D * f_c(C) * dt2(t0) * pass_lik)
  policy_gain_c3a  = sum(f_alpha(t0) * D * f_c(C) * dt3(t0) * pass_lik)
  # electoral benefits
  policy_fail_c1 = sum(R * f_c(C) * dt1(t0) * discount_e)
  policy_fail_c2 = sum(R * f_c(C) * dt2(t0) * discount_e)
  policy_fail_c3 = sum(R * f_c(C) * dt3(t0) * discount_e)
  # overall payoff
  payoff_c1 = policy_gain_c1 + policy_fail_c1 + policy_gain_c1a
  payoff_c2 = policy_gain_c2 + policy_fail_c2 + policy_gain_c2a
  payoff_c3 = policy_gain_c3 + policy_fail_c3 + policy_gain_c3a
  # store payoffs
  payoff_m1s = c(payoff_m1s,payoff_m1)
  payoff_m2s = c(payoff_m2s,payoff_m2)
  payoff_m3s = c(payoff_m3s,payoff_m3)
  payoff_c1s = c(payoff_c1s,payoff_c1)
  payoff_c2s = c(payoff_c2s,payoff_c2)
  payoff_c3s = c(payoff_c3s,payoff_c3)
}

# Figure A13
pdf("game_simulation1_b500.pdf",width = 8,height = 6)
plot(Tss,payoff_m1s,type = "l",ylim = c(-100,800), lty = "dotted", lwd = 2.5,col = "black",
     xlab = "Scrutiny Intensity (by Days)",
     ylab = "Payoff",yaxes = "n")
lines(Tss,payoff_m2s,lty = "dashed", lwd = 2.5,col = "black")
lines(Tss,payoff_m3s,lty = "solid", lwd = 2.5,col = "black")
lines(Tss,payoff_c1s,col = "gray", lty = "dotted", lwd = 2.5)
lines(Tss,payoff_c2s,col = "gray", lty = "dashed", lwd = 2.5)
lines(Tss,payoff_c3s,col = "gray", lty = "solid", lwd = 2.5)
abline(v = 190,lwd = 1, lty = "dashed")
abline(v = 400,lwd = 1, lty = "dashed")
text(170, 145, "Payoff of the Minister")
text(170, 500, "Payoff of the Coalition Partner")
text(100,-100, "Equilibrium Exists")
text(280,-100, "No Equilibrium")
text(450,-100, "Equilibrium Exists")
legend("topleft", legend = c("Constant Timinig", "Early Initiation","Postponing"), lty = c("dotted","dashed","solid"),
       title = "Timing Strategy",lwd = c(2,2,2))
dev.off()


#################################
# Low passage rate in the next term
# Simulation 1: alpha = 0.8
# Low likelihood of bill passage in the next term
pass_lik = 0.1
# Reduced electoral benefit of the minister
discount_e = 1- pass_lik
# Whole term
Period = 4* 365 # days
# Position taking benefits
B = 1
# minister loss from scrutiny
L = 0.5 * B
# loss from policy failure
R = 2 * B
# electoral benefits 
alpha = 0.8
# scrutiny payoff for coalition partner
D = 1
# scrutiny cost
C = 0.8*D
# scrutiny duration
T_s = 365/4
# simulated scrutiny duration
Tss = 0:250
# simulated time space
t = 1: Period
# payoffs
payoff_m1s = c()
payoff_m2s = c()
payoff_m3s = c()
payoff_c1s = c()
payoff_c2s = c()
payoff_c3s = c()
# simulations
for (i in 1:length(Tss)){
  T_s = Tss[i]
  t0 = t[which(Period - t < T_s)]
  t1 = t[which(t< Period - T_s)]
  #####################
  # Payoff of minister
  # policy gain
  policy_gain1 = sum(f_alpha(t1) * dt1(t1) * B)
  policy_gain2 = sum(f_alpha(t1) * dt2(t1) * B)
  policy_gain3 = sum(f_alpha(t1) * dt3(t1) * B)
  # additional policy gain
  policy_gain1a = sum(f_alpha(t0) * dt1(t0) * B * pass_lik)
  policy_gain2a = sum(f_alpha(t0) * dt2(t0) * B * pass_lik)
  policy_gain3a = sum(f_alpha(t0) * dt3(t0) * B * pass_lik)
  # scrutiny cost
  scrutiny_cost1 = sum(L * f_c(C) * dt1(t))
  scrutiny_cost2 = sum(L * f_c(C) * dt2(t))
  scrutiny_cost3 = sum(L * f_c(C) * dt3(t))
  # electoral benefits
  policy_elec1 = sum(R * f_c(C) * dt1(t0) * discount_e)
  policy_elec2 = sum(R * f_c(C) * dt2(t0) * discount_e)
  policy_elec3 = sum(R * f_c(C) * dt3(t0) * discount_e)
  # overall payoff
  payoff_m1 = policy_gain1 - scrutiny_cost1 + policy_elec1 + policy_gain1a
  payoff_m2 = policy_gain2 - scrutiny_cost2 + policy_elec2 + policy_gain2a
  payoff_m3 = policy_gain3 - scrutiny_cost3 + policy_elec3 + policy_gain3a
  # Payoff of coalition partner
  # policy gains from scrutiny
  policy_gain_c1  = sum(f_alpha(t1) * D * f_c(C) * dt1(t1))
  policy_gain_c2  = sum(f_alpha(t1) * D * f_c(C) * dt2(t1))
  policy_gain_c3  = sum(f_alpha(t1) * D * f_c(C) * dt3(t1))
  # additional policy gains 
  policy_gain_c1a  = sum(f_alpha(t0) * D * f_c(C) * dt1(t0) * pass_lik)
  policy_gain_c2a  = sum(f_alpha(t0) * D * f_c(C) * dt2(t0) * pass_lik)
  policy_gain_c3a  = sum(f_alpha(t0) * D * f_c(C) * dt3(t0) * pass_lik)
  # electoral benefits
  policy_fail_c1 = sum(R * f_c(C) * dt1(t0) * discount_e)
  policy_fail_c2 = sum(R * f_c(C) * dt2(t0) * discount_e)
  policy_fail_c3 = sum(R * f_c(C) * dt3(t0) * discount_e)
  # overall payoff
  payoff_c1 = policy_gain_c1 + policy_fail_c1 + policy_gain_c1a
  payoff_c2 = policy_gain_c2 + policy_fail_c2 + policy_gain_c2a
  payoff_c3 = policy_gain_c3 + policy_fail_c3 + policy_gain_c3a
  # store payoffs
  payoff_m1s = c(payoff_m1s,payoff_m1)
  payoff_m2s = c(payoff_m2s,payoff_m2)
  payoff_m3s = c(payoff_m3s,payoff_m3)
  payoff_c1s = c(payoff_c1s,payoff_c1)
  payoff_c2s = c(payoff_c2s,payoff_c2)
  payoff_c3s = c(payoff_c3s,payoff_c3)
}

# Figure A12
pdf("game_simulation1_c.pdf",width = 8,height = 6)
plot(Tss,payoff_m1s,type = "l",ylim = c(-100,800), lty = "dotted", lwd = 2.5,col = "black",
     xlab = "Scrutiny Intensity (by Days)",
     ylab = "Payoff",yaxes = "n")
lines(Tss,payoff_m2s,lty = "dashed", lwd = 2.5,col = "black")
lines(Tss,payoff_m3s,lty = "solid", lwd = 2.5,col = "black")
lines(Tss,payoff_c1s,col = "gray", lty = "dotted", lwd = 2.5)
lines(Tss,payoff_c2s,col = "gray", lty = "dashed", lwd = 2.5)
lines(Tss,payoff_c3s,col = "gray", lty = "solid", lwd = 2.5)
abline(v = 95,lwd = 1, lty = "dashed")
abline(v = 180,lwd = 1, lty = "dashed")
text(170, 145, "Payoff of the Minister")
text(170, 700, "Payoff of the Coalition Partner")
text(40,-100, "Equilibrium Exists")
text(130,-100, "No Equilibrium")
text(210,-100, "Equilibrium Exists")
legend("topleft", legend = c("Constant Timinig", "Early Initiation","Postponing"), lty = c("dotted","dashed","solid"),
       title = "Timing Strategy",lwd = c(2,2,2))
dev.off()

# Total running time
end.time = Sys.time()
time.taken = end.time - start.time
time.taken

