############################################################################################
# Replication File for Supplementary Material 
# Article: "Strategic Postponement of Coalition Policymaking in European Parliamentary Democracies"
# Author: Xiao Lu
# European Journal of Political Research
# 2024
############################################################################################
rm(list=ls())
################
# load packages
library(stargazer)
library(circglmbayes)
library(dplyr)
library(coda)
library(ggplot2)
library(ggrepel)
library(ggthemes)
library(xtable)
#######################
# LOAD MAIN DATASET
load("dat_raw.RData")
dat1 = subset(dat_raw,! dat_raw$country %in% c("Netherlands"))
# Calculate running time
start.time = Sys.time()

#############################
# DESCRIPTIVE STATISTICS
# Table B1 in the appendix
tmp = dat1[,c("bill_location", "month2ne",
              "high_experienced_scrutiny",
              "high_coalition_conflict",
              "high_saliency",
              "min_parl_perc",
              "oppos_conflict",
              "gov_duration",
              "minority",
              "median_oppos")]
stargazer(tmp, 
          covariate.labels = c("Bill Initiation Timing", "Months Remaining", 
                               "High Exp. Scrutiny",
                               "High Coal. Pol. Divergence",  
                               "High Policy Saliency",  
                               "Minister's Party Size",  
                               "Opposition Policy Divergence", 
                               "Government Duration",
                               "Minority Coalition",
                               "Median Opposition"),
          digits=2, 
          summary.stat = c("mean", "sd", "min", "max", "n"),type = "text")

#######################################################
## SUPPLEMENTARY MATERIAL C
# Comparison of explanatory powers 
# Koenig et al 2022:
# Load the data from Koenig et al 2022
load("main_circular_data.RData")
# Delete Netherlands cases
dat_sub = main_circular[which(main_circular$country!="Netherlands"),]
# Model 1 (Hypothesis 1)
set.seed(2345)
model_1k <- circGLM(bill_temporal_location ~ experienced_scrutiny + coalition_policy_divergence + 
                      policy_saliency + minister_party_size + opposition_conflict + 
                      government_duration, 
                    data = dat_sub, Q = 10000, burnin = 1000)
print(model_1k)
# DIC:  85089.6 
# WAIC: 85091.53 
# Model 2 (Hypothesis 2)
set.seed(2345)
model_2k <- circGLM(bill_temporal_location ~ experienced_scrutiny + coalition_policy_divergence + 
                      policy_saliency + minister_party_size + opposition_conflict + government_duration +
                      coalition_policy_divergence*experienced_scrutiny, 
                    data = dat_sub, Q = 10000, burnin = 1000)
print(model_2k)
# DIC:  84931.21 
# WAIC: 84932.57 
# Model 3 (Hypothesis 3)
set.seed(2345)
model_3k <- circGLM(bill_temporal_location ~ experienced_scrutiny + coalition_policy_divergence + 
                      policy_saliency + minister_party_size + opposition_conflict + government_duration +
                      minister_median_party +
                      experienced_scrutiny*minister_party_size*minister_median_party, 
                    data = dat_sub, Q = 10000, burnin = 1000)
print(model_3k)
# DIC:  84934.63 
# WAIC: 84942.9 
######################################
# OWN ANALYSIS:
set.seed(12345)
# Model 1
model_1 <- circGLM(bill_location ~ month2ne, data = dat1, Q = 10000, burnin = 1000)
print(model_1)
# Model 2
set.seed(12345)
model_2 <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne, data = dat1, Q = 10000, burnin = 1000)
print(model_2)
# Model 3
set.seed(12345)
model_3 <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne +
                     high_coalition_conflict + high_saliency + 
                     min_parl_perc +
                     oppos_conflict + gov_duration+
                     minority + median_oppos, data = dat1, Q = 10000, burnin = 1000, thin = 1)
print(model_3)
# Model 4
set.seed(12345)
model_4 <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne +
                     high_coalition_conflict*month2ne + high_saliency*month2ne + 
                     min_parl_perc +
                     oppos_conflict + gov_duration+
                     minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_4)

###############################
## RESULTS OF TABLE C1
# R squared of the models
p1_k = predict(model_1k)
p2_k = predict(model_2k)
p3_k = predict(model_3k)
p1 = predict(model_1)
p2 = predict(model_2)
p3 = predict(model_3)
p4 = predict(model_4)
r2_1k = var(p1_k)/var(dat_sub$bill_temporal_location)
r2_2k = var(p2_k)/var(dat_sub$bill_temporal_location)
r2_3k = var(p3_k)/var(dat_sub$bill_temporal_location)
r2_1 = var(p1)/var(dat1$bill_location)
r2_2 = var(p2)/var(dat1$bill_location)
r2_3 = var(p3)/var(dat1$bill_location)
r2_4 = var(p4)/var(dat1$bill_location)
# Adjusted R2
n = 23936
k = model_1k$n_par
adj_r2_1k = 1 - ((1-r2_1k)*(n-1)/(n-k-1))
k = model_2k$n_par
adj_r2_2k = 1 - ((1-r2_2k)*(n-1)/(n-k-1))
k = model_3k$n_par
adj_r2_3k = 1 - ((1-r2_3k)*(n-1)/(n-k-1))
k = model_1$n_par
adj_r2_1 = 1 - ((1-r2_1)*(n-1)/(n-k-1))
k = model_2$n_par
adj_r2_2 = 1 - ((1-r2_2)*(n-1)/(n-k-1))
k = model_3$n_par
adj_r2_3 = 1 - ((1-r2_3)*(n-1)/(n-k-1))
k = model_4$n_par
adj_r2_4 = 1 - ((1-r2_4)*(n-1)/(n-k-1))
# Show the values
adj_r2_1k # 0.3024709
adj_r2_2k # 0.3402308
adj_r2_3k # 0.317597
adj_r2_1 # 0.887132
adj_r2_2 # 0.8952407
adj_r2_3 # 0.8953296
adj_r2_4 # 0.8950536
# calculate the average values
mean(adj_r2_1k,
     adj_r2_2k,
     adj_r2_3k)
# 0.3024709
mean(adj_r2_1,
     adj_r2_2,
     adj_r2_3,
     adj_r2_4)
# 0.887132
var(p1_k)
var(p1)

# Show the results
print(model_1k)
# DIC:  85089.6 
# WAIC: 85091.53 
print(model_2k)
# DIC:  84931.21 
# WAIC: 84932.57 
print(model_3k)
# DIC:  84934.63 
# WAIC: 84942.9 
print(model_1)
# DIC:  23529.29 
# WAIC: 23531.21
print(model_2)
# DIC:  22512.28 
# WAIC: 22514.25 
print(model_3)
# DIC:  21287.94 
# WAIC: 21288.94 
print(model_4)
# DIC:  21228.91 
# WAIC: 21230.19 

sd(c(predict(model_1k)-dat1$bill_location,
     predict(model_2k)-dat1$bill_location,
     predict(model_3k)-dat1$bill_location))
sd(c(predict(model_1)-dat1$bill_location,
     predict(model_2)-dat1$bill_location,
     predict(model_3)-dat1$bill_location,
     predict(model_4)-dat1$bill_location))

# Mean absolute error (MAE)
# Models without controlling for time budget
sum(abs(predict(model_1k)-dat1$bill_location)) # 33078.33
sum(abs(predict(model_2k)-dat1$bill_location)) # 33281.35
sum(abs(predict(model_3k)-dat1$bill_location)) # 33025.57
# Models controlling for time budget
sum(abs(predict(model_1)-dat1$bill_location)) # 6758.072
sum(abs(predict(model_2)-dat1$bill_location)) # 6628.709
sum(abs(predict(model_3)-dat1$bill_location)) # 6483.218
sum(abs(predict(model_4)-dat1$bill_location)) # 6469.523

# SSE
# Models without controlling for time budget
sum((predict(model_1k)-dat1$bill_location)^2) # 67006.01
sum((predict(model_2k)-dat1$bill_location)^2) # 67956.85
sum((predict(model_3k)-dat1$bill_location)^2) # 66838.35
# Models controlling for time budget
sum((predict(model_1)-dat1$bill_location)^2) # 4432.305
sum((predict(model_2)-dat1$bill_location)^2) # 4160.937
sum((predict(model_3)-dat1$bill_location)^2) # 3931.174
sum((predict(model_4)-dat1$bill_location)^2) # 3915.011

# DIC improvement
# Models controlling for time budget
mean(model_1$DIC,
     model_2$DIC,
     model_3$DIC,
     model_4$DIC
)
# 23529.29
# Models without controlling for time budget
mean(model_1k$DIC,
     model_2k$DIC,
     model_3k$DIC
)
# 85089.6
(1/23529.29 - 1/85089.6)/(1/85089.6)
# 262%

# WAIC improvement
# Models controlling for time budget
mean(model_1$WAIC1,
     model_2$WAIC1,
     model_3$WAIC1,
     model_4$WAIC1
)
# 23531.21
# Models without controlling for time budget
mean(model_1k$WAIC1,
     model_2k$WAIC1,
     model_3k$WAIC1
)
# 85091.53
(1/23531 - 1/85091)/(1/85091) # 2.616123
# 262%

# SSE improvement
mean(sum((predict(model_1)-dat1$bill_location)^2), 
     sum((predict(model_2)-dat1$bill_location)^2), 
     sum((predict(model_3)-dat1$bill_location)^2), 
     sum((predict(model_4)-dat1$bill_location)^2))
# 4432.305
mean(sum((predict(model_1k)-dat1$bill_location)^2), 
     sum((predict(model_2k)-dat1$bill_location)^2), 
     sum((predict(model_3k)-dat1$bill_location)^2))
# 67006.01
(1/4432 - 1/67006)/(1/67006) # 14.11868

# MAE improvement
mean(sum(abs(predict(model_1)-dat1$bill_location)),
     sum(abs(predict(model_2)-dat1$bill_location)),
     sum(abs(predict(model_3)-dat1$bill_location)),
     sum(abs(predict(model_4)-dat1$bill_location)))
# 6758.072
mean(sum(abs(predict(model_1k)-dat1$bill_location)),
     sum(abs(predict(model_2k)-dat1$bill_location)),
     sum(abs(predict(model_3k)-dat1$bill_location)))
# 33078.33
(1/6758 - 1/33078)/(1/33078) # 3.894643

#############################################
# FIGURE C1
# prediction error
e11 = predict(model_1k)-dat1$bill_location
e12 = predict(model_2k)-dat1$bill_location
e13 = predict(model_3k)-dat1$bill_location
e21 = predict(model_1)-dat1$bill_location
e22 = predict(model_2)-dat1$bill_location
e23 = predict(model_3)-dat1$bill_location
e24 = predict(model_4)-dat1$bill_location
model = rep(c("Model 1","Model 2","Model 3","Model 1","Model 2","Model 3","Model 4"),each = 23936)
type = c(rep(c("Without Time Budget","With Time Budget"),each = 3*23936),rep("With Time Budget",each = 23936))
es = c(e11,e12,e13,e21,e22,e23,e24)
d_pred_error = as.data.frame(cbind(type,model,es))
names(d_pred_error)
d_pred_error$es = as.numeric(d_pred_error$es)
d_pred_error$type = factor(d_pred_error$type,levels = c("Without Time Budget","With Time Budget"))
pdf("predict_error.pdf",width = 5,height = 3.5)
ggplot(d_pred_error,aes(x = es))+
  geom_histogram(alpha = 0.4,bins = 500,position="identity")+
  facet_grid(~type)+
  theme_bw()+
  theme(legend.position="bottom")+
  guides(fill=guide_legend(title=""))+
  xlab("Prediction Error")+
  ylab("Count")+
  geom_text(mapping = aes(x, y, label=lab),
            data=data.frame(x=0, y=1500, lab=c("Average SD. = 1.67", "Average SD. = 0.41"),
                            type=factor(c("Without Time Budget","With Time Budget"),levels = c("Without Time Budget","With Time Budget"))))
dev.off()  

###################################################
# SUPPLEMENTARY MATERIAL D.1
# Denmark Case: Sessional analysis
dat4 = subset(dat1,dat1$country == "Denmark")
dat4$last_session_year = dat4$neyear - 1
dat4$last_session_date = as.Date(paste(dat4$last_session_year,"-10-1",sep = ""))
dat4_last_session = dat4[which(dat4$start_date > dat4$last_session_date & dat4$start_date < dat4$ne_date),]
dat4_other_sessions = dat4[which(dat4$start_date <= dat4$last_session_date | dat4$start_date >= dat4$ne_date),]
# DENMARK CASES
# LAST SESSION
set.seed(12345)
model_month_den_lastsession <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne, data = dat4_last_session, Q = 10000, burnin = 1000)
print(model_month_den_lastsession)
# OTHER SESSIONS
set.seed(12345)
model_month_den_othersessions <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne, data = dat4_other_sessions, Q = 10000, burnin = 1000)
print(model_month_den_othersessions)
unique(dat4$neyear)
# Generate Table D1
rownames1 = c(
  "Intercept", "Kappa",
  "Months Remaining",
  "High Exp. Scrutiny * Months Remaining",
  "High Exp. Scrutiny"
)
mod23_chains = model_month_den_lastsession$all_chains
mod24_chains = model_month_den_othersessions$all_chains
tmp11 = apply(mod23_chains,2,mean)[1:5]
tmp12 = t(apply(mod23_chains,2,quantile, probs = c(0.025,0.975)))[1:5,]
tmp1 = round(cbind(tmp11,tmp12),3)
rownames(tmp1) = rownames1
tmp21 = apply(mod24_chains,2,mean)[1:5]
tmp22 = t(apply(mod24_chains,2,quantile, probs = c(0.025,0.975)))[1:5,]
tmp2 = round(cbind(tmp21,tmp22),3)
rownames(tmp2) = rownames1
var.names = row.names(tmp1)[rep(1:length(row.names(tmp1)),each = 2)]
var.names[seq(2,length(var.names),by = 2)] = ""
var.names
ci1 = paste("[", tmp1[,2], ",", tmp1[,3],"]",sep  = "")
ci2 = paste("[", tmp2[,2], ",", tmp2[,3],"]",sep  = "")
tmp1 = tmp1[rep(1:length(row.names(tmp1)),each = 2),1]
tmp2 = tmp2[rep(1:length(row.names(tmp2)),each = 2),1]
tmp1[seq(2,length(var.names),by = 2)] = ci1
tmp2[seq(2,length(var.names),by = 2)] = ci2
coef_tb = cbind(tmp1,tmp2)
row.names(coef_tb) = var.names
xtable(coef_tb)

# GENERATE FIGURE
# Netherlands Case
set.seed(12345)
model_month_net <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne, 
                           data = subset(dat_raw,dat_raw$country == "Netherlands"), Q = 10000, burnin = 1000)
print(model_month_net)
mu_pred_fun = function(beta0,beta_d,x_d = NULL,beta_c,x_c){
  vpi = pi
  if (is.null(x_d)){
    mu = beta0 + 2*atan( x_c %*% beta_c )    
  } else{
    mu = beta0 +x_d %*% beta_d + 2*atan( x_c %*% beta_c )    
  }
  return(mu)
}
tmp = mu_pred_fun(beta0 = 0, beta_c = t(model_month_net$all_chains[,4]),x_c = matrix(seq(48,0)), 
                  beta_d = t(model_month_net$all_chains[,5]), x_d = matrix(rep(1,49)))
tmp_mean = apply(tmp,1,mean)
tmp_q95 = apply(tmp,1,quantile, probs = c(0.025,0.975))
tmp_sim = cbind(tmp_mean,t(tmp_q95))
months_sim9 = as.data.frame(tmp_sim/2/pi*48)
names(months_sim9) = c("mean","ql","qu")
months_sim9$months = seq(48,0)
months_sim9$country = "Netherlands"

# FIGURE D1
pdf("netherlands_effect.pdf",width = 5,height = 4)
ggplot(months_sim9,aes(x = months, y = mean))+
  geom_ribbon(aes(ymin = ql, ymax = qu), fill = "grey70") +
  geom_line()+
  geom_abline(slope = 0, intercept = 0, linetype = "dashed")+
  scale_x_reverse(breaks = c(0,12,24,36,48))+
  xlab("Time Budget")+
  ylab("Late Initiation in Month with High Experienced Scrutiny")+
  theme_bw()+
  facet_wrap(vars(country))
dev.off()


#######################################################
# TABLE D2
# Interaction between scrutiny and coalition conflict
set.seed(12345)
model_int_scrutiny_conflict1 <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne * high_coalition_conflict +
                      + high_saliency + 
                     min_parl_perc +
                     oppos_conflict + gov_duration+
                     minority + median_oppos, data = dat1, Q = 10000, burnin = 1000, thin = 1)
print(model_int_scrutiny_conflict1)

#####################################################
# Sensitivity analysis
# number of bills before
sqrt(var(dat1$duration_passed_bill_gov_area))
# sd. dev of all sample: 63
#  1  10  20  30  40  50  60  70  80 90 100
dat1$sim_sd1 = 1/dat1$num_passed_bill_gov_area
dat1$sim_sd2 = 10/dat1$num_passed_bill_gov_area
dat1$sim_sd3 = 20/dat1$num_passed_bill_gov_area
dat1$sim_sd4 = 30/dat1$num_passed_bill_gov_area
dat1$sim_sd5 = 40/dat1$num_passed_bill_gov_area
dat1$sim_sd6 = 50/dat1$num_passed_bill_gov_area
dat1$sim_sd7 = 60/dat1$num_passed_bill_gov_area
dat1$sim_sd8 = 70/dat1$num_passed_bill_gov_area
dat1$sim_sd9 = 80/dat1$num_passed_bill_gov_area
dat1$sim_sd10 = 90/dat1$num_passed_bill_gov_area
dat1$sim_sd11 = 100/dat1$num_passed_bill_gov_area
dat1$sim_sd12 = 500/dat1$num_passed_bill_gov_area
dat1$sim_sd13 = 1000/dat1$num_passed_bill_gov_area
dat1$sim_sd14 = 6000/dat1$num_passed_bill_gov_area
dat1$scrutiny_sim1 = NA
dat1$scrutiny_sim2 = NA
dat1$scrutiny_sim3 = NA
dat1$scrutiny_sim4 = NA
dat1$scrutiny_sim5 = NA
dat1$scrutiny_sim6 = NA
dat1$scrutiny_sim7 = NA
dat1$scrutiny_sim8 = NA
dat1$scrutiny_sim9 = NA
dat1$scrutiny_sim10 = NA
dat1$scrutiny_sim11 = NA
dat1$scrutiny_sim12 = NA
dat1$scrutiny_sim13 = NA
dat1$scrutiny_sim14 = NA
set.seed(12345)
for (i in 1:dim(dat1)[1]){
  dat1$scrutiny_sim1[i] = rnorm(1,mean = dat1$duration_passed_bill_gov_area[i],sd = dat1$sim_sd1[i])  
  dat1$scrutiny_sim2[i] = rnorm(1,mean = dat1$duration_passed_bill_gov_area[i],sd = dat1$sim_sd2[i])  
  dat1$scrutiny_sim3[i] = rnorm(1,mean = dat1$duration_passed_bill_gov_area[i],sd = dat1$sim_sd3[i])  
  dat1$scrutiny_sim4[i] = rnorm(1,mean = dat1$duration_passed_bill_gov_area[i],sd = dat1$sim_sd4[i])  
  dat1$scrutiny_sim5[i] = rnorm(1,mean = dat1$duration_passed_bill_gov_area[i],sd = dat1$sim_sd5[i])  
  dat1$scrutiny_sim6[i] = rnorm(1,mean = dat1$duration_passed_bill_gov_area[i],sd = dat1$sim_sd6[i])  
  dat1$scrutiny_sim7[i] = rnorm(1,mean = dat1$duration_passed_bill_gov_area[i],sd = dat1$sim_sd7[i])  
  dat1$scrutiny_sim8[i] = rnorm(1,mean = dat1$duration_passed_bill_gov_area[i],sd = dat1$sim_sd8[i])  
  dat1$scrutiny_sim9[i] = rnorm(1,mean = dat1$duration_passed_bill_gov_area[i],sd = dat1$sim_sd9[i])  
  dat1$scrutiny_sim10[i] = rnorm(1,mean = dat1$duration_passed_bill_gov_area[i],sd = dat1$sim_sd10[i])  
  dat1$scrutiny_sim11[i] = rnorm(1,mean = dat1$duration_passed_bill_gov_area[i],sd = dat1$sim_sd11[i])  
}
set.seed(12345)
for (i in 1:dim(dat1)[1]){
  dat1$scrutiny_sim12[i] = rnorm(1,mean = dat1$duration_passed_bill_gov_area[i],sd = dat1$sim_sd12[i])  
  dat1$scrutiny_sim13[i] = rnorm(1,mean = dat1$duration_passed_bill_gov_area[i],sd = dat1$sim_sd13[i])  
}  
set.seed(12345)
for (i in 1:dim(dat1)[1]){
  dat1$scrutiny_sim14[i] = rnorm(1,mean = dat1$duration_passed_bill_gov_area[i],sd = dat1$sim_sd14[i])  
}  

dat1$scrutiny_sim1 = ifelse(dat1$scrutiny_sim1<=0,0,dat1$scrutiny_sim1)
dat1$scrutiny_sim2 = ifelse(dat1$scrutiny_sim2<=0,0,dat1$scrutiny_sim2)
dat1$scrutiny_sim3 = ifelse(dat1$scrutiny_sim3<=0,0,dat1$scrutiny_sim3)
dat1$scrutiny_sim4 = ifelse(dat1$scrutiny_sim4<=0,0,dat1$scrutiny_sim4)
dat1$scrutiny_sim5 = ifelse(dat1$scrutiny_sim5<=0,0,dat1$scrutiny_sim5)
dat1$scrutiny_sim6 = ifelse(dat1$scrutiny_sim6<=0,0,dat1$scrutiny_sim6)
dat1$scrutiny_sim7 = ifelse(dat1$scrutiny_sim7<=0,0,dat1$scrutiny_sim7)
dat1$scrutiny_sim8 = ifelse(dat1$scrutiny_sim8<=0,0,dat1$scrutiny_sim8)
dat1$scrutiny_sim9 = ifelse(dat1$scrutiny_sim9<=0,0,dat1$scrutiny_sim9)
dat1$scrutiny_sim10 = ifelse(dat1$scrutiny_sim10<=0,0,dat1$scrutiny_sim10)
dat1$scrutiny_sim11 = ifelse(dat1$scrutiny_sim11<=0,0,dat1$scrutiny_sim11)
dat1$scrutiny_sim12 = ifelse(dat1$scrutiny_sim12<=0,0,dat1$scrutiny_sim12)
dat1$scrutiny_sim13 = ifelse(dat1$scrutiny_sim13<=0,0,dat1$scrutiny_sim13)
dat1$scrutiny_sim14 = ifelse(dat1$scrutiny_sim14<=0,0,dat1$scrutiny_sim14)

dat1 = dat1 %>%
  group_by(country) %>%
  mutate(experienced_scrutiny_country_median_sim1 = median(scrutiny_sim1)) %>%
  mutate(experienced_scrutiny_country_median_sim2 = median(scrutiny_sim2)) %>%
  mutate(experienced_scrutiny_country_median_sim3 = median(scrutiny_sim3)) %>%
  mutate(experienced_scrutiny_country_median_sim4 = median(scrutiny_sim4)) %>%
  mutate(experienced_scrutiny_country_median_sim5 = median(scrutiny_sim5)) %>%
  mutate(experienced_scrutiny_country_median_sim6 = median(scrutiny_sim6)) %>%
  mutate(experienced_scrutiny_country_median_sim7 = median(scrutiny_sim7)) %>%
  mutate(experienced_scrutiny_country_median_sim8 = median(scrutiny_sim8)) %>%
  mutate(experienced_scrutiny_country_median_sim9 = median(scrutiny_sim9)) %>%
  mutate(experienced_scrutiny_country_median_sim10 = median(scrutiny_sim10)) %>%
  mutate(experienced_scrutiny_country_median_sim11 = median(scrutiny_sim11)) %>%
  as.data.frame()

dat1 = dat1 %>%
  group_by(country) %>%
  mutate(experienced_scrutiny_country_median_sim12 = median(scrutiny_sim12)) %>%
  mutate(experienced_scrutiny_country_median_sim13 = median(scrutiny_sim13)) %>%
  as.data.frame()

dat1 = dat1 %>%
  group_by(country) %>%
  mutate(experienced_scrutiny_country_median_sim14 = median(scrutiny_sim14)) %>%
  as.data.frame()

dat1$high_experienced_scrutiny_sim1 = ifelse(dat1$scrutiny_sim1 > dat1$experienced_scrutiny_country_median_sim1,1,0)
dat1$high_experienced_scrutiny_sim2 = ifelse(dat1$scrutiny_sim2 > dat1$experienced_scrutiny_country_median_sim2,1,0)
dat1$high_experienced_scrutiny_sim3 = ifelse(dat1$scrutiny_sim3 > dat1$experienced_scrutiny_country_median_sim3,1,0)
dat1$high_experienced_scrutiny_sim4 = ifelse(dat1$scrutiny_sim4 > dat1$experienced_scrutiny_country_median_sim4,1,0)
dat1$high_experienced_scrutiny_sim5 = ifelse(dat1$scrutiny_sim5 > dat1$experienced_scrutiny_country_median_sim5,1,0)
dat1$high_experienced_scrutiny_sim6 = ifelse(dat1$scrutiny_sim6 > dat1$experienced_scrutiny_country_median_sim6,1,0)
dat1$high_experienced_scrutiny_sim7 = ifelse(dat1$scrutiny_sim7 > dat1$experienced_scrutiny_country_median_sim7,1,0)
dat1$high_experienced_scrutiny_sim8 = ifelse(dat1$scrutiny_sim8 > dat1$experienced_scrutiny_country_median_sim8,1,0)
dat1$high_experienced_scrutiny_sim9 = ifelse(dat1$scrutiny_sim9 > dat1$experienced_scrutiny_country_median_sim9,1,0)
dat1$high_experienced_scrutiny_sim10 = ifelse(dat1$scrutiny_sim10 > dat1$experienced_scrutiny_country_median_sim10,1,0)
dat1$high_experienced_scrutiny_sim11 = ifelse(dat1$scrutiny_sim11 > dat1$experienced_scrutiny_country_median_sim11,1,0)
dat1$high_experienced_scrutiny_sim12 = ifelse(dat1$scrutiny_sim12 > dat1$experienced_scrutiny_country_median_sim12,1,0)
dat1$high_experienced_scrutiny_sim13 = ifelse(dat1$scrutiny_sim13 > dat1$experienced_scrutiny_country_median_sim13,1,0)
dat1$high_experienced_scrutiny_sim14 = ifelse(dat1$scrutiny_sim14 > dat1$experienced_scrutiny_country_median_sim14,1,0)

set.seed(12345)
model_month_sim1 <- circGLM(bill_location ~ high_experienced_scrutiny_sim1 * month2ne +
                              high_coalition_conflict*month2ne + high_saliency*month2ne + 
                              min_parl_perc +
                              oppos_conflict + gov_duration+
                              minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_sim1)
set.seed(12345)
model_month_sim2 <- circGLM(bill_location ~ high_experienced_scrutiny_sim2 * month2ne +
                              high_coalition_conflict*month2ne + high_saliency*month2ne + 
                              min_parl_perc +
                              oppos_conflict + gov_duration+
                              minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_sim2)
set.seed(12345)
model_month_sim3 <- circGLM(bill_location ~ high_experienced_scrutiny_sim3 * month2ne +
                              high_coalition_conflict*month2ne + high_saliency*month2ne + 
                              min_parl_perc +
                              oppos_conflict + gov_duration+
                              minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_sim3)
set.seed(12345)
model_month_sim4 <- circGLM(bill_location ~ high_experienced_scrutiny_sim4 * month2ne +
                              high_coalition_conflict*month2ne + high_saliency*month2ne + 
                              min_parl_perc +
                              oppos_conflict + gov_duration+
                              minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_sim4)
set.seed(12345)
model_month_sim5 <- circGLM(bill_location ~ high_experienced_scrutiny_sim5 * month2ne +
                              high_coalition_conflict*month2ne + high_saliency*month2ne + 
                              min_parl_perc +
                              oppos_conflict + gov_duration+
                              minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_sim5)
set.seed(12345)
model_month_sim6 <- circGLM(bill_location ~ high_experienced_scrutiny_sim6 * month2ne +
                              high_coalition_conflict*month2ne + high_saliency*month2ne + 
                              min_parl_perc +
                              oppos_conflict + gov_duration+
                              minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_sim6)
set.seed(12345)
model_month_sim7 <- circGLM(bill_location ~ high_experienced_scrutiny_sim7 * month2ne +
                              high_coalition_conflict*month2ne + high_saliency*month2ne + 
                              min_parl_perc +
                              oppos_conflict + gov_duration+
                              minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_sim7)
set.seed(12345)
model_month_sim8 <- circGLM(bill_location ~ high_experienced_scrutiny_sim8 * month2ne +
                              high_coalition_conflict*month2ne + high_saliency*month2ne + 
                              min_parl_perc +
                              oppos_conflict + gov_duration+
                              minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_sim8)
set.seed(12345)
model_month_sim9 <- circGLM(bill_location ~ high_experienced_scrutiny_sim9 * month2ne +
                              high_coalition_conflict*month2ne + high_saliency*month2ne + 
                              min_parl_perc +
                              oppos_conflict + gov_duration+
                              minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_sim9)
set.seed(12345)
model_month_sim10 <- circGLM(bill_location ~ high_experienced_scrutiny_sim10 * month2ne +
                               high_coalition_conflict*month2ne + high_saliency*month2ne + 
                               min_parl_perc +
                               oppos_conflict + gov_duration+
                               minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_sim10)
set.seed(12345)
model_month_sim11 <- circGLM(bill_location ~ high_experienced_scrutiny_sim11 * month2ne +
                               high_coalition_conflict*month2ne + high_saliency*month2ne + 
                               min_parl_perc +
                               oppos_conflict + gov_duration+
                               minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_sim11)
set.seed(12345)
model_month_sim12 <- circGLM(bill_location ~ high_experienced_scrutiny_sim12 * month2ne +
                               high_coalition_conflict*month2ne + high_saliency*month2ne + 
                               min_parl_perc +
                               oppos_conflict + gov_duration+
                               minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_sim12)
set.seed(12345)
model_month_sim13 <- circGLM(bill_location ~ high_experienced_scrutiny_sim13 * month2ne +
                               high_coalition_conflict*month2ne + high_saliency*month2ne + 
                               min_parl_perc +
                               oppos_conflict + gov_duration+
                               minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_sim13)
set.seed(12345)
model_month_sim14 <- circGLM(bill_location ~ high_experienced_scrutiny_sim14 * month2ne +
                               high_coalition_conflict*month2ne + high_saliency*month2ne + 
                               min_parl_perc +
                               oppos_conflict + gov_duration+
                               minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_sim14)

# bt_chain5: time-budget * scrutiny; dt_chain.1: scrutiny
apply(model_month_sim14$bt_chain,2,mean)
apply(model_month_sim14$dt_chain,2,mean)

mean_est_scrutiny = c()
mean_est_interaction = c()
q1_est_scrutiny = c()
q2_est_scrutiny = c()
q1_est_interaction = c()
q2_est_interaction = c()

mean_est_scrutiny[1] = mean(model_month_sim1$dt_chain[,1])
mean_est_scrutiny[2] = mean(model_month_sim2$dt_chain[,1])
mean_est_scrutiny[3] = mean(model_month_sim3$dt_chain[,1])
mean_est_scrutiny[4] = mean(model_month_sim4$dt_chain[,1])
mean_est_scrutiny[5] = mean(model_month_sim5$dt_chain[,1])
mean_est_scrutiny[6] = mean(model_month_sim6$dt_chain[,1])
mean_est_scrutiny[7] = mean(model_month_sim7$dt_chain[,1])
mean_est_scrutiny[8] = mean(model_month_sim8$dt_chain[,1])
mean_est_scrutiny[9] = mean(model_month_sim9$dt_chain[,1])
mean_est_scrutiny[10] = mean(model_month_sim10$dt_chain[,1])
mean_est_scrutiny[11] = mean(model_month_sim11$dt_chain[,1])
mean_est_scrutiny[12] = mean(model_month_sim12$dt_chain[,1])
mean_est_scrutiny[13] = mean(model_month_sim13$dt_chain[,1])
mean_est_scrutiny[14] = mean(model_month_sim14$dt_chain[,1])

mean_est_interaction[1] = mean(model_month_sim1$bt_chain[,5])
mean_est_interaction[2] = mean(model_month_sim2$bt_chain[,5])
mean_est_interaction[3] = mean(model_month_sim3$bt_chain[,5])
mean_est_interaction[4] = mean(model_month_sim4$bt_chain[,5])
mean_est_interaction[5] = mean(model_month_sim5$bt_chain[,5])
mean_est_interaction[6] = mean(model_month_sim6$bt_chain[,5])
mean_est_interaction[7] = mean(model_month_sim7$bt_chain[,5])
mean_est_interaction[8] = mean(model_month_sim8$bt_chain[,5])
mean_est_interaction[9] = mean(model_month_sim9$bt_chain[,5])
mean_est_interaction[10] = mean(model_month_sim10$bt_chain[,5])
mean_est_interaction[11] = mean(model_month_sim11$bt_chain[,5])
mean_est_interaction[12] = mean(model_month_sim12$bt_chain[,5])
mean_est_interaction[13] = mean(model_month_sim13$bt_chain[,5])
mean_est_interaction[14] = mean(model_month_sim14$bt_chain[,5])

q1_est_scrutiny[1] = quantile(model_month_sim1$dt_chain[,1],probs = 0.025)
q1_est_scrutiny[2] = quantile(model_month_sim2$dt_chain[,1],probs = 0.025)
q1_est_scrutiny[3] = quantile(model_month_sim3$dt_chain[,1],probs = 0.025)
q1_est_scrutiny[4] = quantile(model_month_sim4$dt_chain[,1],probs = 0.025)
q1_est_scrutiny[5] = quantile(model_month_sim5$dt_chain[,1],probs = 0.025)
q1_est_scrutiny[6] = quantile(model_month_sim6$dt_chain[,1],probs = 0.025)
q1_est_scrutiny[7] = quantile(model_month_sim7$dt_chain[,1],probs = 0.025)
q1_est_scrutiny[8] = quantile(model_month_sim8$dt_chain[,1],probs = 0.025)
q1_est_scrutiny[9] = quantile(model_month_sim9$dt_chain[,1],probs = 0.025)
q1_est_scrutiny[10] = quantile(model_month_sim10$dt_chain[,1],probs = 0.025)
q1_est_scrutiny[11] = quantile(model_month_sim11$dt_chain[,1],probs = 0.025)
q1_est_scrutiny[12] = quantile(model_month_sim12$dt_chain[,1],probs = 0.025)
q1_est_scrutiny[13] = quantile(model_month_sim13$dt_chain[,1],probs = 0.025)
q1_est_scrutiny[14] = quantile(model_month_sim14$dt_chain[,1],probs = 0.025)

q2_est_scrutiny[1] = quantile(model_month_sim1$dt_chain[,1],probs = 0.975)
q2_est_scrutiny[2] = quantile(model_month_sim2$dt_chain[,1],probs = 0.975)
q2_est_scrutiny[3] = quantile(model_month_sim3$dt_chain[,1],probs = 0.975)
q2_est_scrutiny[4] = quantile(model_month_sim4$dt_chain[,1],probs = 0.975)
q2_est_scrutiny[5] = quantile(model_month_sim5$dt_chain[,1],probs = 0.975)
q2_est_scrutiny[6] = quantile(model_month_sim6$dt_chain[,1],probs = 0.975)
q2_est_scrutiny[7] = quantile(model_month_sim7$dt_chain[,1],probs = 0.975)
q2_est_scrutiny[8] = quantile(model_month_sim8$dt_chain[,1],probs = 0.975)
q2_est_scrutiny[9] = quantile(model_month_sim9$dt_chain[,1],probs = 0.975)
q2_est_scrutiny[10] = quantile(model_month_sim10$dt_chain[,1],probs = 0.975)
q2_est_scrutiny[11] = quantile(model_month_sim11$dt_chain[,1],probs = 0.975)
q2_est_scrutiny[12] = quantile(model_month_sim12$dt_chain[,1],probs = 0.975)
q2_est_scrutiny[13] = quantile(model_month_sim13$dt_chain[,1],probs = 0.975)
q2_est_scrutiny[14] = quantile(model_month_sim14$dt_chain[,1],probs = 0.975)

q1_est_interaction[1] = quantile(model_month_sim1$bt_chain[,5],probs = 0.025)
q1_est_interaction[2] = quantile(model_month_sim2$bt_chain[,5],probs = 0.025)
q1_est_interaction[3] = quantile(model_month_sim3$bt_chain[,5],probs = 0.025)
q1_est_interaction[4] = quantile(model_month_sim4$bt_chain[,5],probs = 0.025)
q1_est_interaction[5] = quantile(model_month_sim5$bt_chain[,5],probs = 0.025)
q1_est_interaction[6] = quantile(model_month_sim6$bt_chain[,5],probs = 0.025)
q1_est_interaction[7] = quantile(model_month_sim7$bt_chain[,5],probs = 0.025)
q1_est_interaction[8] = quantile(model_month_sim8$bt_chain[,5],probs = 0.025)
q1_est_interaction[9] = quantile(model_month_sim9$bt_chain[,5],probs = 0.025)
q1_est_interaction[10] = quantile(model_month_sim10$bt_chain[,5],probs = 0.025)
q1_est_interaction[11] = quantile(model_month_sim11$bt_chain[,5],probs = 0.025)
q1_est_interaction[12] = quantile(model_month_sim12$bt_chain[,5],probs = 0.025)
q1_est_interaction[13] = quantile(model_month_sim13$bt_chain[,5],probs = 0.025)
q1_est_interaction[14] = quantile(model_month_sim14$bt_chain[,5],probs = 0.025)

q2_est_interaction[1] = quantile(model_month_sim1$bt_chain[,5],probs = 0.975)
q2_est_interaction[2] = quantile(model_month_sim2$bt_chain[,5],probs = 0.975)
q2_est_interaction[3] = quantile(model_month_sim3$bt_chain[,5],probs = 0.975)
q2_est_interaction[4] = quantile(model_month_sim4$bt_chain[,5],probs = 0.975)
q2_est_interaction[5] = quantile(model_month_sim5$bt_chain[,5],probs = 0.975)
q2_est_interaction[6] = quantile(model_month_sim6$bt_chain[,5],probs = 0.975)
q2_est_interaction[7] = quantile(model_month_sim7$bt_chain[,5],probs = 0.975)
q2_est_interaction[8] = quantile(model_month_sim8$bt_chain[,5],probs = 0.975)
q2_est_interaction[9] = quantile(model_month_sim9$bt_chain[,5],probs = 0.975)
q2_est_interaction[10] = quantile(model_month_sim10$bt_chain[,5],probs = 0.975)
q2_est_interaction[11] = quantile(model_month_sim11$bt_chain[,5],probs = 0.975)
q2_est_interaction[12] = quantile(model_month_sim12$bt_chain[,5],probs = 0.975)
q2_est_interaction[13] = quantile(model_month_sim13$bt_chain[,5],probs = 0.975)
q2_est_interaction[14] = quantile(model_month_sim14$bt_chain[,5],probs = 0.975)

est_scrutiny = as.data.frame(cbind(seq = 1:14, sims = c(1,10,20,30,40,50,60,70,80,90,100,500,1000,6000),
                                   mean = mean_est_scrutiny,q1 = q1_est_scrutiny,q2 = q2_est_scrutiny))

est_interaction = as.data.frame(cbind(seq = 1:14, sims = c(1,10,20,30,40,50,60,70,80,90,100,500,1000,6000),
                                      mean = mean_est_interaction,q1 = q1_est_interaction,q2 = q2_est_interaction))
# FIGURE D2 (a)
pdf("robust_sd_scrutiny.pdf",width = 4.75,height = 4.2)
ggplot(est_scrutiny,aes(x = seq,y = mean))+
  geom_line()+
  geom_ribbon(aes(ymin = q1,ymax = q2),alpha = 0.4)+
  theme_bw()+
  scale_x_continuous(breaks=0:14,
                     labels=c(0,1,10,20,30,40,50,60,70,80,90,100,500,1000,6000),name = "Standard Deviation")+
  ylab("Estimated Effects of High Exp. Scrutiny")+
  geom_vline(xintercept = 7.3,linetype = "dotted",size = 1.2)+
  geom_text(x=7.02, y=0.11, label="Sample Standard Deviation = 63")+
  ylim(c(0,0.32))+
  geom_point(x = 0, y = 0.292,col = "red")+
  geom_segment(aes(x = 0, y = 0.274, xend = 0, yend = 0.310),col = "red")+
  geom_segment(aes(x = 0, y = 0.27, xend = 1, yend = 0.22),size = 0.3)+
  geom_text(x = 3, y = 0.2,label = "Original Estimate")
dev.off()
# FIGURE D2 (b)
pdf("robust_sd_interaction.pdf",width = 4.75,height = 4.2)
ggplot(est_interaction,aes(x = seq,y = mean))+
  geom_line()+
  geom_ribbon(aes(ymin = q1,ymax = q2),alpha = 0.4)+
  theme_bw()+
  scale_x_continuous(breaks=0:14,
                     labels=c(0,1,10,20,30,40,50,60,70,80,90,100,500,1000,6000),name = "Standard Deviation")+
  ylab("Estimated Effects: High Exp. Scrutiny * Time Budget")+
  geom_vline(xintercept = 7.3,linetype = "dotted",size = 1.2)+
  geom_text(x=7.02, y=-0.02, label="Sample Standard Deviation = 63")+
  ylim(c(-0.076,0))+
  geom_point(x = 0, y = -0.062,col = "red")+
  geom_segment(aes(x = 0, y = -0.070, xend = 0, yend = -0.055),col = "red")+
  geom_segment(aes(x = 0, y = -0.052, xend = 1, yend = -0.04),size = 0.3)+
  geom_text(x = 3, y = -0.037,label = "Original Estimate")
dev.off()



######################################################
# Interaction:policy divergence * saliency
# aggregated estimates: policy divergence * saliency
# use only residual values of coalition divergence and saliency
e1 = glm(high_coalition_conflict ~ high_experienced_scrutiny,data = dat1,family = binomial(link = "logit")) #
dat1$conflict_residual = ifelse(e1$residuals>=0,1,0)
e2 = glm(high_saliency ~ duration_passed_bill_gov_area,data = dat1,family = binomial(link = "logit"))
dat1$saliency_residual = ifelse(e2$residuals>=0,1,0)
cor(dat1$high_saliency,dat1$high_experienced_scrutiny)
cor(dat1$high_coalition_conflict,dat1$high_experienced_scrutiny)

# Table D3
set.seed(12345)
model_month_int = circGLM(formula = bill_location ~ high_experienced_scrutiny * 
                            month2ne + high_coalition_conflict * month2ne + high_saliency * 
                            month2ne + high_coalition_conflict * high_saliency * 
                            month2ne +
                            + min_parl_perc + oppos_conflict + gov_duration + 
                            minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_month_int)

# Table D4
set.seed(12345)
model_month_residual1 <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne +
                                   conflict_residual*month2ne + saliency_residual*month2ne + 
                                   min_parl_perc +
                                   oppos_conflict + gov_duration+
                                   minority + median_oppos, data = dat1, Q = 1000, burnin = 1000)
print(model_month_residual1)

########################################################
# Interactions with expected government next term
# Approximated by real government composition next term
# Table D5
set.seed(12345)
model_con1 <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne +
                        high_experienced_scrutiny * return_share + 
                        high_experienced_scrutiny * party_in_nxt_gov +
                     high_coalition_conflict*month2ne + high_saliency*month2ne + 
                     min_parl_perc +
                     oppos_conflict + gov_duration+
                     minority + median_oppos, data = dat1, Q = 10000, burnin = 1000)
print(model_con1)

#########################################################
### Robustness checks
# Table D6
# Country-fixed effects
set.seed(12345)
model_month_01_c <- circGLM(bill_location ~ month2ne + country, data = dat1, Q = 10000, burnin = 1000)
print(model_month_01_c)
set.seed(12345)
model_month_11_c <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne + country, data = dat1, Q = 10000, burnin = 1000)
print(model_month_11_c)
set.seed(12345)
model_month_21_c <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne +
                              high_coalition_conflict + high_saliency + 
                              min_parl_perc +
                              oppos_conflict + gov_duration+
                              minority + median_oppos + country, data = dat1, Q = 10000, burnin = 1000, thin = 1)
print(model_month_21_c)
set.seed(12345)
model_month_22_c <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne +
                              high_coalition_conflict*month2ne + high_saliency*month2ne + 
                              min_parl_perc +
                              oppos_conflict + gov_duration+
                              minority + median_oppos + country, data = dat1, Q = 10000, burnin = 1000)
print(model_month_22_c)

# Table D7
# Policy-fixed effects
set.seed(12345)
model_month_01_p <- circGLM(bill_location ~ month2ne + area13_f, data = dat1, Q = 10000, burnin = 1000)
print(model_month_01_p) 
set.seed(12345)
model_month_11_p <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne + area13_f, data = dat1, Q = 10000, burnin = 1000)
print(model_month_11_p) 
set.seed(12345)
model_month_21_p <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne +
                              high_coalition_conflict + high_saliency + 
                              min_parl_perc +
                              oppos_conflict + gov_duration+
                              minority + median_oppos + area13_f, data = dat1, Q = 10000, burnin = 1000, thin = 1)
print(model_month_21_p) 
set.seed(12345)
model_month_22_p <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne +
                              high_coalition_conflict*month2ne + high_saliency*month2ne + 
                              min_parl_perc +
                              oppos_conflict + gov_duration+
                              minority + median_oppos + area13_f, data = dat1, Q = 10000, burnin = 1000)
print(model_month_22_p) 

# Table D8
# Two-way Policy-country-fixed effects
set.seed(12345)
model_month_01_pc <- circGLM(bill_location ~ month2ne + area13_f + country, data = dat1, Q = 10000, burnin = 1000)
print(model_month_01_pc) 
set.seed(12345)
model_month_11_pc <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne + area13_f + country, data = dat1, Q = 10000, burnin = 1000)
print(model_month_11_pc) 
set.seed(12345)
model_month_21_pc <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne +
                               high_coalition_conflict + high_saliency + 
                               min_parl_perc +
                               oppos_conflict + gov_duration+
                               minority + median_oppos + area13_f + country, data = dat1, Q = 10000, burnin = 1000, thin = 1)
print(model_month_21_pc) 
set.seed(12345)
model_month_22_pc <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne +
                               high_coalition_conflict*month2ne + high_saliency*month2ne + 
                               min_parl_perc +
                               oppos_conflict + gov_duration+
                               minority + median_oppos + area13_f + country, data = dat1, Q = 10000, burnin = 1000)
print(model_month_22_pc) 

# Table D9
# One-way Policy-country-fixed effects
dat1$country_policy = paste(dat1$country,dat1$area13_f,sep = "-")
set.seed(12345)
model_month_01_pc1 <- circGLM(bill_location ~ month2ne + country_policy, data = dat1, Q = 10000, burnin = 1000)
print(model_month_01_pc1) 
set.seed(12345)
model_month_11_pc1 <- circGLM(bill_location ~ high_experienced_scrutiny*month2ne + country_policy, data = dat1, Q = 10000, burnin = 1000)
print(model_month_11_pc1) 
set.seed(12345)
model_month_21_pc1 <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne +
                               high_coalition_conflict + high_saliency + 
                               min_parl_perc +
                               oppos_conflict + gov_duration+
                               minority + median_oppos + country_policy, data = dat1, Q = 10000, burnin = 1000, thin = 1)
print(model_month_21_pc1) 
set.seed(12345)
model_month_22_pc1 <- circGLM(bill_location ~ high_experienced_scrutiny * month2ne +
                               high_coalition_conflict*month2ne + high_saliency*month2ne + 
                               min_parl_perc +
                               oppos_conflict + gov_duration+
                               minority + median_oppos + country_policy, data = dat1, Q = 10000, burnin = 1000)
print(model_month_22_pc1) 

# Total running time
end.time = Sys.time()
time.taken = end.time - start.time
time.taken


  