# ---- Analysis 2: Individual and contextual changes are weakly related to changes in trust ----

# ---- Within-between models ----

# Regression models

# Limit to larger amount of waves

comp_liss <- liss %>% filter(confpolt != "NA") %>% group_by(id) %>% mutate(count=n()) %>% filter(count>=12) %>% ungroup()
comp_bes <- bes %>% filter(trustMPs != "NA") %>% group_by(id) %>% mutate(count=n()) %>% filter(count >= 13) %>% ungroup()
comp_shp <- shp %>% filter(pp04 != "NA") %>% group_by(id)  %>% mutate(count=n()) %>% filter(count>=10) %>% ungroup()

# Change to factor variables 

comp_liss <- comp_liss %>% mutate(
  workstatus = factor(workstatus, levels = c(1, 2, 3, 4)),
  edu3 = factor(edu3, levels = c(1, 2, 3)),
  polint = factor(polint, levels = c(1, 2, 3)))

comp_bes <- comp_bes %>% mutate(
  workstatus = factor(workstatus, levels = c(1, 2, 3, 4)),
  edu3 = factor(edu3, levels = c(1, 2, 3)),
  polint = factor(polint, levels = c(1, 2, 3)))

comp_shp <- comp_shp %>% mutate(
  workstatus = factor(workstatus, levels = c(1, 2, 3, 4)),
  edu3 = factor(edu3, levels = c(1, 2, 3)),
  polint = factor(polint, levels = c(1, 2, 3)))

comp_liss$confpolt_std <- (comp_liss$confpolt - mean(comp_liss$confpolt)) / sd(comp_liss$confpolt)

comp_bes$trustMPs_std <- (comp_bes$trustMPs - mean(comp_bes$trustMPs)) / sd(comp_bes$trustMPs)

comp_shp$pp04_std <- (comp_shp$pp04 - mean(comp_shp$pp04)) / sd(comp_shp$pp04)

# De-mean all the relevant variables to obtain within-between estimates

d_liss <- cbind(
  comp_liss,
  demean(comp_liss, select = c("income10", "edu3", 
                               "workstatus", "leftright",
                               "polint", "age"), group = "id") # from package "parameters"
)

d_shp <- cbind(
  comp_shp,
  demean(comp_shp, select = c("income10", "edu3", 
                              "workstatus", "leftright",
                              "polint", "age"), group = "id") # from package "parameters"
)

d_bes <- cbind(
  comp_bes,
  demean(comp_bes, select = c("income10", "edu3", 
                              "workstatus", "leftright",
                              "polint", "age"), group = "id") # from package "parameters"
)

# Change so standardise all between 0 and 1

d_bes <- d_bes %>%
  mutate(trustMPs = (trustMPs-1)/6)

d_liss <- d_liss %>%
  mutate(confpolt = confpolt/10)

d_shp <- d_shp %>%
  mutate(pp04 = pp04/10)

# Run models, save estimates/CIs 

model_rewb_liss <- lmer( # the intention is to run a multilevel model of change with a random slope on time to let individuals have different over-time change
  confpolt ~ age_within + age_between + 
    income10_within + income10_between + 
    edu3_2_within + edu3_2_between +
    edu3_3_within + edu3_3_between + 
    workstatus_2_within + workstatus_2_between +
    workstatus_3_within + workstatus_3_between +
    workstatus_4_within + workstatus_4_between +
    leftright_within + leftright_between + 
    polint_2_within + polint_2_between +
    polint_3_within + polint_3_between +
    gender + wave +
    (1 + wave | id),
  data = d_liss,
  control = lmerControl(check.nobs.vs.nRE = "ignore")
)

liss_estimates <- as.data.frame(getME(model_rewb_liss, "fixef")) %>% tibble::rownames_to_column() %>% setNames(c("variable","estimate")) %>% mutate(
  wb = case_when(str_detect(variable, "within") == "TRUE" ~ "within",
                 str_detect(variable, "between") == "TRUE" ~ "between",
                 TRUE ~ "time invariant"),
  variable = gsub("_within", "", variable),
  variable = gsub("_between", "", variable))

liss_ci <- as.data.frame(confint(model_rewb_liss)) %>% tibble::rownames_to_column() %>% setNames(c("variable", "lower", "upper")) %>%
  mutate(
    wb = case_when(str_detect(variable, "within") == "TRUE" ~ "within",
                   str_detect(variable, "between") == "TRUE" ~ "between",
                   TRUE ~ "time invariant"),
    variable = gsub("_within", "", variable),
    variable = gsub("_between", "", variable))

liss_estimates <- full_join(liss_estimates, liss_ci)
liss_estimates$panel <- "LISS"

model_rewb_shp <- lmer(
  pp04 ~ age_within + age_between + 
    income10_within + income10_between + 
    edu3_2_within + edu3_2_between +
    edu3_3_within + edu3_3_between + 
    workstatus_2_within + workstatus_2_between +
    workstatus_3_within + workstatus_3_between +
    workstatus_4_within + workstatus_4_between +
    leftright_within + leftright_between + 
    polint_2_within + polint_2_between +
    polint_3_within + polint_3_between +
    gender + wave +
    (1 + wave | id) + (1|idhous),
  data = d_shp,
  control = lmerControl(check.nobs.vs.nRE = "ignore")
)

shp_estimates <- as.data.frame(getME(model_rewb_shp, "fixef")) %>% tibble::rownames_to_column() %>% setNames(c("variable","estimate")) %>% mutate(
  wb = case_when(str_detect(variable, "within") == "TRUE" ~ "within",
                 str_detect(variable, "between") == "TRUE" ~ "between",
                 TRUE ~ "time invariant"),
  variable = gsub("_within", "", variable),
  variable = gsub("_between", "", variable),
  panel = "SHP")

shp_ci <- as.data.frame(confint(model_rewb_shp)) %>% tibble::rownames_to_column() %>% setNames(c("variable", "lower", "upper")) %>%
  mutate(
    wb = case_when(str_detect(variable, "within") == "TRUE" ~ "within",
                   str_detect(variable, "between") == "TRUE" ~ "between",
                   TRUE ~ "time invariant"),
    variable = gsub("_within", "", variable),
    variable = gsub("_between", "", variable))

shp_estimates <- full_join(shp_estimates, shp_ci)
shp_estimates$panel <- "SHP"


model_rewb_bes <- lmer(
  trustMPs ~ age_within + age_between + 
    income10_within + income10_between + 
    edu3_2_within + edu3_2_between +
    edu3_3_within + edu3_3_between + 
    workstatus_2_within + workstatus_2_between +
    workstatus_3_within + workstatus_3_between +
    workstatus_4_within + workstatus_4_between +
    leftright_within + leftright_between + 
    polint_2_within + polint_2_between +
    polint_3_within + polint_3_between +
    gender + wave +
    (1 + wave |id),
  data = d_bes,
  control = lmerControl(check.nobs.vs.nRE = "ignore")
)

bes_estimates <- as.data.frame(getME(model_rewb_bes, "fixef")) %>% tibble::rownames_to_column() %>% setNames(c("variable","estimate")) %>% 
  mutate(
    wb = case_when(str_detect(variable, "within") == "TRUE" ~ "within",
                   str_detect(variable, "between") == "TRUE" ~ "between",
                   TRUE ~ "time invariant"),
    variable = gsub("_within", "", variable),
    variable = gsub("_between", "", variable))

bes_ci <- as.data.frame(confint(model_rewb_bes)) %>% tibble::rownames_to_column() %>% setNames(c("variable", "lower", "upper")) %>%
  mutate(
    wb = case_when(str_detect(variable, "within") == "TRUE" ~ "within",
                   str_detect(variable, "between") == "TRUE" ~ "between",
                   TRUE ~ "time invariant"),
    variable = gsub("_within", "", variable),
    variable = gsub("_between", "", variable))

bes_estimates <- full_join(bes_estimates, bes_ci)
bes_estimates$panel <- "BES"

# Merge all the estimates/CIs and graph
# Figure 4: The within and between effects on political trust in three panel studies

all_estimates <- rbind(bes_estimates, liss_estimates, shp_estimates)

dodge <- position_dodge(width=0.5)
all_estimates %>% filter(wb != "time invariant") %>%
  ggplot(aes(x = estimate, y = variable, colour = wb)) +
  facet_wrap(~panel) +
  geom_point(position = dodge, size = 2) +
  geom_linerange(aes(xmin = lower, xmax = upper, colour = wb), position=dodge, size = 1) +
  theme_minimal() +
  #scale_colour_grey() + 
  scale_y_discrete(labels=c("Age", "Educ: Medium", "Educ: University", "Income", "Left-Right", "Political Interest: Fairly",
                            "Political Interest: Very", "Work: Student", "Work: Retired", "Work: Unemployed/Other")) +
  geom_vline(xintercept = 0, alpha = 0.2) +
  theme(strip.text = element_text(size=12),
        legend.title = element_blank(),
        legend.position="bottom",
        legend.key.size = unit(1, 'cm'),
        legend.text = element_text(size=10),
        axis.text.y = element_text(size = 10),
        axis.title.x=element_text(size=10)) +
  labs(y = "", x = "Estimate") 

ggsave("wbm_mods.png", 
       path = here("figures"))

# Grayscale
# dodge <- position_dodge(width=0.5)
# all_estimates %>% filter(wb != "time invariant") %>%
#   ggplot(aes(x = estimate, y = variable, colour = wb)) +
#   facet_wrap(~panel) +
#   geom_point(position = dodge, size = 2) +
#   geom_linerange(aes(xmin = lower, xmax = upper, colour = wb), position=dodge, size = 1) +
#   theme_minimal() +
#   #scale_colour_grey() + 
#   scale_y_discrete(labels=c("Age", "Educ: Medium", "Educ: University", "Income", "Left-Right", "Political Interest: Fairly",
#                             "Political Interest: Very", "Work: Student", "Work: Retired", "Work: Unemployed/Other")) +
#   geom_vline(xintercept = 0, alpha = 0.2) +
#   theme(strip.text = element_text(size=12),
#         legend.title = element_blank(),
#         legend.position="bottom",
#         legend.key.size = unit(1, 'cm'),
#         legend.text = element_text(size=10),
#         axis.text.y = element_text(size = 10),
#         axis.title.x=element_text(size=10)) +
#   labs(y = "", x = "Estimate") + scale_colour_grey()
# 
# ggsave("wbm_mods.png", 
#        path = here("figures"))

# ---- Impact functions ----
# Figure 5: Impact function of losing an election on average levels of trust

liss_incumbdf <- liss %>% group_by(id) %>%
  # incumbent change - make year if incumbency changes
  mutate(incumbch_yr = ifelse(incumb_ch == -1, year, 0), # year you BECAME a loser
         incumbch_yr = max(incumbch_yr),
         time_ch = year-incumbch_yr) %>% # time to/from loser change
  filter(time_ch < 18) %>% ungroup() %>% select(-incumbch_yr) %>%
  mutate(time_ch_short = ifelse(time_ch %in% c(-18:-3), -3,
                                ifelse(time_ch %in% c(3:18), 3, time_ch)),
         tsqu = time_ch^2,
         toch = ifelse(time_ch < 1, time_ch, 0), # time until change
         fromch = ifelse(time_ch >-1, time_ch, 0), # time from incumbency change
         lost = ifelse(time_ch >-1, 1, 0)) # dummy for having lost


liss_incumbdf2 <- liss %>% group_by(id) %>%
  # incumbent change - make year if incumbency changes
  mutate(incumbch_yr = ifelse(incumb_ch == 1, year, 0), # year you BECAME a loser
         incumbch_yr = max(incumbch_yr),
         time_ch = year-incumbch_yr) %>% # time to/from loser change
  filter(time_ch < 18) %>% ungroup() %>% select(-incumbch_yr)

impact_df <- liss_incumbdf %>% 
  select(id, year, lost, fromch, toch, tsqu, time_ch,
         age, income10, edu3, workstatus, gender, leftright, confpolt, vote) %>%
  filter_all(all_vars(!is.na(.))) %>% arrange(id, year)

impact_df <- pdata.frame(impact_df, index = c("id", "year"))
impact_df <- zap_labels(impact_df)

impact_df2 <- liss_incumbdf2 %>% 
  select(id, year, time_ch,
         age, income10, edu3, workstatus, gender, leftright, confpolt, vote) %>%
  filter_all(all_vars(!is.na(.))) %>% arrange(id, year)

impact_df2 <- pdata.frame(impact_df2, index = c("id", "year"))
impact_df2 <- zap_labels(impact_df2)

exmpl <- impact_df %>% filter(time_ch %in% c(-4:4)) %>% 
  group_by(time_ch) %>% summarise(av_trst = mean(confpolt))

exmpl2 <- impact_df2 %>% filter(time_ch %in% c(-4:4)) %>% 
  group_by(time_ch) %>% summarise(av_trst = mean(confpolt))

exmpl$status <- "Election loser"
exmpl2$status <- "Election winner"

impactfun <- rbind(exmpl, exmpl2)

plot <- ggplot(impactfun, aes(x = time_ch, y = av_trst)) + 
  geom_line(aes(color = status), size = 1) +
  scale_x_discrete(limits = c(-4:4)) +
  geom_vline(xintercept = 0, linetype="dashed") +
  theme_minimal() +
  labs(x = "Time (in years) to/since changing incumbency status",
       y = "Average confidence in politicians (0-10)",
       caption = "Data: LISS Panel Study")+
  annotate("text", x = -1, y=6.565, label = "Turnover", 
           colour="black", fontface=2) +
  annotate(
    geom = "curve", x = -1, y = 6.53, xend = 0, yend = 6.45, 
    curvature = .3, arrow = arrow(length = unit(2, "mm")), colour="black") +
  theme(legend.title=element_blank()) +
  annotate("text", x = -3, y=5.35, label = "Election winner", 
           colour="#00BFC4", fontface=2) +
  annotate("text", x = -2, y=6.5, label = "Election loser", 
           colour="#F8766D", fontface=2) +
  theme(legend.position="none")

ggsave("winlose.png", 
       path = here("figures"))
