####################
# Replication code for Wratil, Wäckerle: "Majority Representation and Legitimacy: Survey-Experimental Evidence from the European Union" ####
# Appendix ####
# General Information: All plots in the paper and the Appendix are done using 
# the tikzDevice package. This means that plots from this script displayed 
# in the preview window or saved to pdf/png might look slightly different, but
# only in terms of size of axis labels, headings and the like. The quantities 
# displayed are equivalent to the ones in the paper/appendix.

#### Intro ####

library(cregg)
library(tidyverse)
library(dotwhisker)
library(scales)
library(gridExtra)
get_legend<-function(myggplot){
  tmp <- ggplot_gtable(ggplot_build(myggplot))
  leg <- which(sapply(tmp$grobs, function(x) x$name) == "guide-box")
  legend <- tmp$grobs[[leg]]
  return(legend)
}

load("datasets_eu.RData")

label_changes=c("<blank>" = "No",
                "In this instance, the European Commission asked EU citizens and interest groups to submit their views on the topic." = "Yes",
                "All national governments voted in favor." = "Unanimity",
                "Most national governments voted in favor including the [COUNTRY] government, some voted against the decision." = "Majority incl. government",
                "Most national governments voted in favor, some voted against the decision including the [COUNTRY] government." = "Majority not incl. government",
                "Most" = "Large majority",
                "Slightly more than half" = "Slight majority",
                "Most EU citizens oppose this decision" = "No",
                "Most EU citizens support this decision" = "Yes",
                " most [NATIONALITY] support it" = "Yes",
                " most [NATIONALITY] oppose it" = "No")

label_changes_topics=c("<blank>" = "No",
                       "Most EU citizens support this decision, most [NATIONALITY] support it" = "EU support, national support",
                       "Most EU citizens oppose this decision, most [NATIONALITY] support it" = "EU oppose, national support",
                       "Most EU citizens support this decision, most [NATIONALITY] oppose it" = "EU support, national oppose",
                       "Most EU citizens oppose this decision, most [NATIONALITY] oppose it" = "EU oppose, national oppose")

theme_demos <- theme(plot.title = element_text(face="bold"),
                     axis.text=element_text(colour="black"),
                     axis.title=element_text(size=8),
                     strip.text.y = element_text(angle = 0, face="bold"),
                     strip.background = element_blank(),
                     panel.border = element_blank(), 
                     axis.ticks = element_blank(),
                     panel.background = element_blank(), 
                     panel.spacing.x = unit(0,"line"),
                     legend.title = element_blank())

theme_voting <-   theme(plot.title = element_text(face="bold"),
                        axis.title=element_text(size=8),
                        strip.text.y = element_text(angle = 0, face="bold"),
                        strip.background = element_blank(),
                        panel.border = element_blank(), 
                        axis.ticks = element_blank(),
                        panel.background = element_blank(), 
                        panel.spacing.x = unit(0,"line"),
                        legend.position = "bottom",
                        legend.title = element_blank(),
                        axis.text.x = element_text(angle = 45, hjust = 1))

theme_results <-  theme(plot.title = element_text(face="bold"),
                        axis.text=element_text(colour="black"),
                        axis.title=element_text(size=8,colour="black"),
                        strip.text.y = element_text(angle = 0, face="bold"),
                        strip.background = element_blank(),
                        panel.border = element_blank(), 
                        axis.ticks = element_blank(),
                        panel.background = element_blank(), 
                        panel.spacing.x = unit(0,"line"),
                        legend.position = "bottom",
                        legend.title = element_blank())

#### Appendix B ####
#### Table B1 ####

load(file="pretest.RData")

cor(pretest_scales_all %>% select(-vignette))
pretest_scales_all <- pretest_scales_all%>%rename("Q1_legitimate"="legitimate",
                                       "Q2_accept"="accept",
                                       "Q3_should_right"="should_right",
                                       "Q4_political_action"="political_action",
                                       "Q5_implement"="implement",
                                       "Q6_legal"="legal",
                                       "EU_member"="eu_membership",
                                       "Trust_in_EU"="trust_eu")
# This is Table B1
print(xtable::xtable(cor(pretest_scales_all %>% select(-vignette))))

#### Appendix C ####
#### Figure C1 ####
eudata$country_name <- recode(eudata$country_code,
                              "ES" = "Spain",
                              "IT" = "Italy",
                              "DE" = "Germany",
                              "FR" = "France",
                              "PL" = "Poland")
age.plot <- ggplot(eudata, aes(age)) +
  geom_density(adjust=2,size=2) +
  xlab("Age")+
  ylab("Density")+
  theme_bw()+
  ggtitle("Age of Respondents")+
  facet_wrap(~country_name)+
  theme_demos

sex.plot <- ggplot(eudata, aes(group = country_code, x=gender)) +
  geom_bar(aes(y=..prop..),stat="count")+
  xlab("")+
  #  scale_y_continuous(labels=scales::percent) +
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("Sex of Respondents")+
  facet_wrap(~country_name)+
  theme_demos

# This is Figure C1
ggpubr::ggarrange(age.plot,sex.plot)

#### Figure C2 ####

urban.plot <- ggplot(eudata, aes(group = country_code, x=city_or_rural)) +
  geom_bar(aes(y=..prop..),stat="count")+
  xlab("")+
  #  scale_y_continuous(labels=scales::percent) +
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("City and Rural")+
  facet_wrap(~country_name)+
  theme_demos

eudata$education_level=factor(eudata$education_level,levels = c("no","low", "medium", "high"))

plot.education <- ggplot(eudata, aes(group = country_code, x=education_level)) +
  geom_bar(aes(y=..prop..),stat="count")+
  xlab("")+
  #  scale_y_continuous(labels=scales::percent) +
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("Education of Respondents")+
  facet_wrap(~country_name)+
  theme_demos

# This is Figure C2
ggpubr::ggarrange(urban.plot,plot.education)

#### Appendix D ####
#### Figure D1 ####
load(file="eubaro_lire.RData")

eurobarometer_lire_germany=eurobaro_90_1 %>% 
  filter(!is.na(left_right)&country=="Germany" & q1_3 == "Mentioned") %>%
  select(left_right)
eurobarometer_lire_germany$method="Eurobarometer"

eurobarometer_lire_germany$left_right=rescale(eurobarometer_lire_germany$left_right,to=c(0,10))
eudata_lire_germany=data.frame=eudata %>% filter(country_code=="DE")%>%select(lire)
names(eudata_lire_germany)="left_right"
eudata_lire_germany$method="Survey"
germany_lire=rbind(eurobarometer_lire_germany,eudata_lire_germany)

plot_lire_germany=ggplot(germany_lire, aes(left_right, colour = method,lty=method)) +
  geom_density(adjust=2,size=2) +
  xlab("Left-Right")+
  ylab("Density")+
  theme_bw()+
  ggtitle("b) Germany")+
  scale_colour_grey(start = 0, end = .5) +
  theme_demos

eurobarometer_lire_italy=eurobaro_90_1 %>% 
  filter(!is.na(left_right)&country=="Italy" & q1_8 == "Mentioned") %>%
  select(left_right)
eurobarometer_lire_italy$method="Eurobarometer"
eurobarometer_lire_italy$left_right=rescale(eurobarometer_lire_italy$left_right,to=c(0,10))
eudata_lire_italy=data.frame=eudata %>% filter(country_code=="IT")%>%select(lire)
names(eudata_lire_italy)="left_right"
eudata_lire_italy$method="Survey"
italy_lire=rbind(eurobarometer_lire_italy,eudata_lire_italy)

plot_lire_italy=ggplot(italy_lire, aes(left_right, colour = method,lty=method)) +
  geom_density(adjust=2,size=2) +
  xlab("Left-Right")+
  ylab("Density")+
  theme_bw()+
  ggtitle("c) Italy")+
  scale_colour_grey(start = 0, end = .5) +
  theme_demos

eurobarometer_lire_spain=eurobaro_90_1 %>% 
  filter(!is.na(left_right)&country=="Spain" & q1_5 == "Mentioned") %>%
  select(left_right)
eurobarometer_lire_spain$method="Eurobarometer"
eurobarometer_lire_spain$left_right=rescale(eurobarometer_lire_spain$left_right,to=c(0,10))
eudata_lire_spain=data.frame=eudata %>% filter(country_code=="ES")%>%select(lire)
names(eudata_lire_spain)="left_right"
eudata_lire_spain$method="Survey"
spain_lire=rbind(eurobarometer_lire_spain,eudata_lire_spain)

plot_lire_spain=ggplot(spain_lire, aes(left_right, colour = method,lty=method)) +
  geom_density(adjust=2,size=2) +
  xlab("Left-Right")+
  ylab("Density")+
  theme_bw()+
  ggtitle("d) Spain")+
  scale_colour_grey(start = 0, end = .5) +
  theme_demos

eurobarometer_lire_poland=eurobaro_90_1 %>% 
  filter(!is.na(left_right)&country=="Poland" & q1_23 == "Mentioned") %>%
  select(left_right)
eurobarometer_lire_poland$method="Eurobarometer"
eurobarometer_lire_poland$left_right=rescale(eurobarometer_lire_poland$left_right,to=c(0,10))
eudata_lire_poland=data.frame=eudata %>% filter(country_code=="PL")%>%select(lire)
names(eudata_lire_poland)="left_right"
eudata_lire_poland$method="Survey"
poland_lire=rbind(eurobarometer_lire_poland,eudata_lire_poland)

plot_lire_poland=ggplot(poland_lire, aes(left_right, colour = method,lty=method)) +
  geom_density(adjust=2,size=2) +
  xlab("Left-Right")+
  ylab("Density")+
  theme_bw()+
  ggtitle("e) Poland")+
  scale_colour_grey(start = 0, end = .5)+
  theme_demos

eurobarometer_lire_france=eurobaro_90_1 %>% 
  filter(!is.na(left_right)&country=="France" & q1_6 == "Mentioned") %>%
  select(left_right)
eurobarometer_lire_france$method="Eurobarometer"
eurobarometer_lire_france$left_right=rescale(eurobarometer_lire_france$left_right,to=c(0,10))
eudata_lire_france=data.frame=eudata %>% filter(country_code=="FR")%>%select(lire)
names(eudata_lire_france)="left_right"
eudata_lire_france$method="Survey"
france_lire=rbind(eurobarometer_lire_france,eudata_lire_france)

plot_lire_france=ggplot(france_lire, aes(left_right, colour = method,lty=method)) +
  geom_density(adjust=2,size=2) +
  xlab("Left-Right")+
  ylab("Density")+
  theme_bw()+
  ggtitle("a) France")+
  scale_colour_grey(start = 0, end = .5) +
  theme_demos

legend <- get_legend(plot_lire_france)
plot_lire_france <- plot_lire_france + theme(legend.position="none")
plot_lire_germany <- plot_lire_germany + theme(legend.position="none")
plot_lire_italy <- plot_lire_italy + theme(legend.position="none")
plot_lire_spain <- plot_lire_spain + theme(legend.position="none")
plot_lire_poland <- plot_lire_poland + theme(legend.position="none")

# This is Figure D1
gridExtra::grid.arrange(plot_lire_france,plot_lire_germany,plot_lire_italy,
                        plot_lire_poland,plot_lire_spain,legend,ncol=2)

#### Figure D2 ####

eudata$EU_membership = recode(eudata$EU_membership,
                              "A bad thing" = "A bad \nthing",
                              "A good thing" = "A good \nthing",
                              "Neither a good thing nor a bad thing" = "Neither")

load(file="eubaro_member.RData")

eurobarometer_member_germany=eurobaro_90_1 %>% 
  filter(!is.na(eu_membership)&country=="Germany" & q1_3 == "Mentioned") %>%
  select(eu_membership)
eurobarometer_member_germany$method="Eurobarometer"
eudata_member_germany=data.frame=eudata %>% filter(country_code=="DE")%>%select(EU_membership)
names(eudata_member_germany)="eu_membership"
eudata_member_germany$method="Survey"
germany_member=rbind(eurobarometer_member_germany,eudata_member_germany)

aggregate(germany_member$eu_membership,by=list(germany_member$method),FUN=summary)
table(germany_member$eu_membership)
plot_member_germany=ggplot(germany_member, aes(group = method, x=eu_membership,fill=method)) +
  geom_bar(aes(y=..prop..),stat="count",position="dodge")+
  ylim(c(0,0.8))+
  xlab("")+
  scale_fill_grey(start = 0, end = .5) +
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("b) Germany")+
  theme_demos

eurobarometer_member_italy=eurobaro_90_1 %>% 
  filter(!is.na(eu_membership)&country=="Italy" & q1_8 == "Mentioned") %>%
  select(eu_membership)
eurobarometer_member_italy$method="Eurobarometer"
eudata_member_italy=data.frame=eudata %>% filter(country_code=="IT")%>%select(EU_membership)
names(eudata_member_italy)="eu_membership"
eudata_member_italy$method="Survey"
italy_member=rbind(eurobarometer_member_italy,eudata_member_italy)

plot_member_italy=ggplot(italy_member, aes(group = method, x=eu_membership,fill=method)) +
  geom_bar(aes(y=..prop..),stat="count",position="dodge")+
  ylim(c(0,0.8))+
  xlab("")+
  scale_fill_grey(start = 0, end = .5) +
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("c) Italy")+
  theme_demos

eurobarometer_member_spain=eurobaro_90_1 %>% 
  filter(!is.na(eu_membership)&country=="Spain" & q1_5 == "Mentioned") %>%
  select(eu_membership)
eurobarometer_member_spain$method="Eurobarometer"
eudata_member_spain=data.frame=eudata %>% filter(country_code=="ES")%>%select(EU_membership)
names(eudata_member_spain)="eu_membership"
eudata_member_spain$method="Survey"
spain_member=rbind(eurobarometer_member_spain,eudata_member_spain)

plot_member_spain=ggplot(spain_member, aes(group = method, x=eu_membership,fill=method)) +
  geom_bar(aes(y=..prop..),stat="count",position="dodge")+
  ylim(c(0,0.8))+
  xlab("")+
  scale_fill_grey(start = 0, end = .5) +
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("d) Spain")+
  theme_demos

eurobarometer_member_poland=eurobaro_90_1 %>% 
  filter(!is.na(eu_membership)&country=="Poland" & q1_23 == "Mentioned") %>%
  select(eu_membership)
eurobarometer_member_poland$method="Eurobarometer"
eudata_member_poland=data.frame=eudata %>% filter(country_code=="PL")%>%select(EU_membership)
names(eudata_member_poland)="eu_membership"
eudata_member_poland$method="Survey"
poland_member=rbind(eurobarometer_member_poland,eudata_member_poland)

plot_member_poland=ggplot(poland_member, aes(group = method, x=eu_membership,fill=method)) +
  geom_bar(aes(y=..prop..),stat="count",position="dodge")+
  xlab("")+
  ylim(c(0,0.8))+
  scale_fill_grey(start = 0, end = .5) +
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("e) Poland")+
  theme_demos

eurobarometer_member_france=eurobaro_90_1 %>% 
  filter(!is.na(eu_membership)&country=="France" & q1_6 == "Mentioned") %>%
  select(eu_membership)
eurobarometer_member_france$method="Eurobarometer"
eudata_member_france=data.frame=eudata %>% filter(country_code=="FR")%>%select(EU_membership)
names(eudata_member_france)="eu_membership"
eudata_member_france$method="Survey"
france_member=rbind(eurobarometer_member_france,eudata_member_france)

plot_member_france=ggplot(france_member, aes(group = method, x=eu_membership,fill=method)) +
  geom_bar(aes(y=..prop..),stat="count",position="dodge")+
  xlab("")+
  scale_fill_grey(start = 0, end = .5) +
  ylim(c(0,0.8))+
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("a) France")+
  theme_demos

legend <- get_legend(plot_member_france)
plot_member_france <- plot_member_france + theme(legend.position="none")
plot_member_germany <- plot_member_germany + theme(legend.position="none")
plot_member_italy <- plot_member_italy + theme(legend.position="none")
plot_member_spain <- plot_member_spain + theme(legend.position="none")
plot_member_poland <- plot_member_poland + theme(legend.position="none")

# This is Figure D2
gridExtra::grid.arrange(plot_member_france,plot_member_germany,plot_member_italy,
                        plot_member_poland,plot_member_spain,legend,ncol=2)

#### Figure D3 ####

eudata$EU_identity[eudata$EU_identity=="None of the above"]="None or \nDon't \nknow"

load(file="eubaro_identity.RData")

eurobarometer_identity_germany=eurobaro_90_3 %>% 
  filter(!is.na(eu_identity)&country=="Germany" & q1_3 == "Mentioned") %>%
  select(eu_identity)
eurobarometer_identity_germany$method="Eurobarometer"
eudata_identity_germany=data.frame=eudata %>% filter(country_code=="DE")%>%select(EU_identity)
names(eudata_identity_germany)="eu_identity"
eudata_identity_germany$method="Survey"
germany_identity=rbind(eurobarometer_identity_germany,eudata_identity_germany)

plot_identity_germany=ggplot(germany_identity, aes(group = method, x=eu_identity,fill=method)) +
  geom_bar(aes(y=..prop..),stat="count",position="dodge")+
  xlab("")+
  ylim(c(0,0.8))+
  scale_fill_grey(start = 0, end = .5) +
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("b) Germany")+
  theme_demos

eurobarometer_identity_italy=eurobaro_90_3 %>% 
  filter(!is.na(eu_identity)&country=="Italy" & q1_8 == "Mentioned") %>%
  select(eu_identity)
eurobarometer_identity_italy$method="Eurobarometer"
eudata_identity_italy=data.frame=eudata %>% filter(country_code=="IT")%>%select(EU_identity)
names(eudata_identity_italy)="eu_identity"
eudata_identity_italy$method="Survey"
italy_identity=rbind(eurobarometer_identity_italy,eudata_identity_italy)

plot_identity_italy=ggplot(italy_identity, aes(group = method, x=eu_identity,fill=method)) +
  geom_bar(aes(y=..prop..),stat="count",position="dodge")+
  xlab("")+
  scale_fill_grey(start = 0, end = .5) +
  ylim(c(0,0.8))+
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("c) Italy")+
  theme_demos

eurobarometer_identity_spain=eurobaro_90_3 %>% 
  filter(!is.na(eu_identity)&country=="Spain" & q1_5 == "Mentioned") %>%
  select(eu_identity)
eurobarometer_identity_spain$method="Eurobarometer"
eudata_identity_spain=data.frame=eudata %>% filter(country_code=="ES")%>%select(EU_identity)
names(eudata_identity_spain)="eu_identity"
eudata_identity_spain$method="Survey"
spain_identity=rbind(eurobarometer_identity_spain,eudata_identity_spain)

plot_identity_spain=ggplot(spain_identity, aes(group = method, x=eu_identity,fill=method)) +
  geom_bar(aes(y=..prop..),stat="count",position="dodge")+
  xlab("")+
  scale_fill_grey(start = 0, end = .5) +
  ylim(c(0,0.8))+
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("d) Spain")+
  theme_demos

eurobarometer_identity_poland=eurobaro_90_3 %>% 
  filter(!is.na(eu_identity)&country=="Poland" & q1_23 == "Mentioned") %>%
  select(eu_identity)
eurobarometer_identity_poland$method="Eurobarometer"
eudata_identity_poland=data.frame=eudata %>% filter(country_code=="PL")%>%select(EU_identity)
names(eudata_identity_poland)="eu_identity"
eudata_identity_poland$method="Survey"
poland_identity=rbind(eurobarometer_identity_poland,eudata_identity_poland)

plot_identity_poland=ggplot(poland_identity, aes(group = method, x=eu_identity,fill=method)) +
  geom_bar(aes(y=..prop..),stat="count",position="dodge")+
  xlab("")+
  scale_fill_grey(start = 0, end = .5) +
  ylim(c(0,0.8))+
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("e) Poland")+
  theme_demos

eurobarometer_identity_france=eurobaro_90_3 %>% 
  filter(!is.na(eu_identity)&country=="France" & q1_6 == "Mentioned") %>%
  select(eu_identity)
eurobarometer_identity_france$method="Eurobarometer"
eudata_identity_france=data.frame=eudata %>% filter(country_code=="FR")%>%select(EU_identity)
names(eudata_identity_france)="eu_identity"
eudata_identity_france$method="Survey"
france_identity=rbind(eurobarometer_identity_france,eudata_identity_france)

plot_identity_france=ggplot(france_identity, aes(group = method, x=eu_identity,fill=method)) +
  geom_bar(aes(y=..prop..),stat="count",position="dodge")+
  xlab("")+
  scale_fill_grey(start = 0, end = .5) +
  ylim(c(0,0.8))+
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("a) France")+
  theme_demos

legend <- get_legend(plot_identity_france)
plot_identity_france <- plot_identity_france + theme(legend.position="none")
plot_identity_germany <- plot_identity_germany + theme(legend.position="none")
plot_identity_italy <- plot_identity_italy + theme(legend.position="none")
plot_identity_spain <- plot_identity_spain + theme(legend.position="none")
plot_identity_poland <- plot_identity_poland + theme(legend.position="none")

# This is Figure D3
gridExtra::grid.arrange(plot_identity_france,plot_identity_germany,plot_identity_italy,
                        plot_identity_poland,plot_identity_spain,legend,ncol=2)

#### Figure D4 ####

voting_germany=data.frame(result=c(eudata%>%
                                     filter(!vote_de%in%c("","Don't know","Would not vote"))%>%
                                     select(vote_de)%>%
                                     rename(vote=vote_de)),
                          country="Germany")%>%
  within(vote <- factor(vote,
                        levels=names(sort(table(vote),
                                          decreasing = T))))

voting_germany$vote=recode(voting_germany$vote,
                           "Christlich Demokratische Union/Christlich Soziale Union (CDU/CSU)" = "CDU/CSU",
                           "Sozialdemokratische Partei Deutschlands (SPD)" = "SPD",
                           "Freie Demokratische Partei (FDP)" = "FDP",
                           "Alternative für Deutschland (AfD)" = "AfD",
                           "Bündnis 90/Die Grünen" = "Die Grünen")

voting_italy=data.frame(result=c(eudata%>%
                                   filter(!vote_it%in%c("","Don't know","Would not vote"))%>%
                                   select(vote_it)%>%
                                   rename(vote=vote_it)),
                        country="Italy")%>%
  within(vote <- factor(vote,
                        levels=names(sort(table(vote),
                                          decreasing = T))))

voting_italy$vote=recode(voting_italy$vote,
                         "Lega (LN)" = "Lega",
                         "Partito Democratico (PD)" = "PD",
                         "Movimento 5 Stelle (M5S)" = "M5S",
                         "Fratelli d'Italia (FdI)" = "FdI",
                         "Potere al Popolo (PaP)" = "PaP",
                         "Forza Italia (FI)" = "FI")

voting_spain=data.frame(result=c(eudata%>%
                                   filter(!vote_es%in%c("","Don't know","Would not vote"))%>%
                                   select(vote_es)%>%
                                   rename(vote=vote_es)),
                        country="Spain")%>%
  within(vote <- factor(vote,
                        levels=names(sort(table(vote),
                                          decreasing = T))))

voting_spain$vote=recode(voting_spain$vote,
                         "Partido Socialista Obrero Español (PSOE)" = "PSOE",
                         "Partido Popular (PP)" = "PP",
                         "Ciudadanos (Cs)" = "Cs",
                         "Unidos Podemos" = "Podemos")

voting_france=data.frame(result=c(eudata%>%
                                    filter(!vote_fr%in%c("","Don't know","Would not vote"))%>%
                                    select(vote_fr)%>%
                                    rename(vote=vote_fr)),
                         country="France")%>%
  within(vote <- factor(vote,
                        levels=names(sort(table(vote),
                                          decreasing = T))))

voting_france$vote=recode(voting_france$vote,
                          "La République En Marche! (LREM)" = "LREM",
                          "Front National (FN)" = "FN",
                          "La France Insoumise (FI)" = "FI",
                          "Les Républicains (LR)" = "LR",
                          "Parti Socialiste (PS)" = "PS",
                          "Mouvement Démocrate (MoDem)" = "MoDem",
                          "Union des Démocrates et Indépendants (UDI)" = "UDI")

voting_poland=data.frame(result=c(eudata%>%
                                    filter(!vote_pl%in%c("","Don't know","Would not vote"))%>%
                                    select(vote_pl)%>%
                                    rename(vote=vote_pl)),
                         country="Poland")%>%
  within(vote <- factor(vote,
                        levels=names(sort(table(vote),
                                          decreasing = T))))

voting_poland$vote=recode(voting_poland$vote,
                          "Kukiz'15 (K'15)" = "K'15",
                          "Polskie Stronnictwo Ludowe (PSL)" = "PSL",
                          "Platformę Obywatelską (PO)" = "PO",
                          "Prawo i Sprawiedliwość (PiS)" = "PiS",
                          ".Nowoczesną (.N)" = ".N",
                          "Sojusz Lewicy Demokratycznej (SLD)" = "SLD")

p_poland=ggplot(voting_poland, aes(x=vote)) +
  geom_bar(aes(y=(..count..)/sum(..count..)),stat="count")+
  xlab("")+
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("d) Poland")+
  theme_voting

p_germany=ggplot(voting_germany, aes(x=vote)) +
  geom_bar(aes(y=(..count..)/sum(..count..)),stat="count")+
  xlab("")+
  #  scale_y_continuous(labels=scales::percent) +
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("b) Germany")+
  theme_voting

p_italy=ggplot(voting_italy, aes(x=vote)) +
  geom_bar(aes(y=(..count..)/sum(..count..)),stat="count")+
  xlab("")+
  #  scale_y_continuous(labels=scales::percent) +
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("c) Italy")+
  theme_voting

p_spain=ggplot(voting_spain, aes(x=vote)) +
  geom_bar(aes(y=(..count..)/sum(..count..)),stat="count")+
  xlab("")+
  #  scale_y_continuous(labels=scales::percent) +
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("e) Spain")+
  theme_voting

p_france=ggplot(voting_france, aes(x=vote)) +
  geom_bar(aes(y=(..count..)/sum(..count..)),stat="count")+
  xlab("")+
  #  scale_y_continuous(labels=scales::percent) +
  ylab("Share of Respondents")+
  theme_bw()+
  ggtitle("a) France")+
  theme_voting

# This is Figure D4
gridExtra::grid.arrange(p_france,p_germany,p_italy,p_poland,p_spain,ncol=2)

#### Appendix E ####
#### Figure E1 ####

# Issue 1: Environment
eudata$issue1_environment_rec=recode(eudata$issue1_environment,
                                     "Increase environmental protection at the cost of more regulation for businesses" = "Increase environmental protection",
                                     "Decrease environmental protection with the benefit of less regulation for businesses" = "Decrease environmental protection")

issue_1_results <- data.frame(table(eudata$issue1_environment_rec,eudata$country_code))

issue_1_data <- data.frame(country_code=unique(eudata$country_code))
issue_1_data$country_code <- as.character(issue_1_data$country_code)
issue_1_data$share_increase <- NA
for(i in 1:nrow(issue_1_data)){
  country <- issue_1_data$country_code[i]
  issue_1_data$share_increase[issue_1_data$country_code == country] <- issue_1_results$Freq[issue_1_results$Var1 == "Increase environmental protection" & issue_1_results$Var2==country]/
    (issue_1_results$Freq[issue_1_results$Var1 == "Increase environmental protection" & issue_1_results$Var2==country]+
       issue_1_results$Freq[issue_1_results$Var1 == "Decrease environmental protection" & issue_1_results$Var2==country])
}

issue_1_data$share_increase <- round(issue_1_data$share_increase,2)
issue_1_data$country_name <- recode(issue_1_data$country_code,
                                    "ES" = "Spain",
                                    "IT" = "Italy",
                                    "DE" = "Germany",
                                    "FR" = "France",
                                    "PL" = "Poland")

plot_issue_1=ggplot(issue_1_data, aes(x = country_name, y=share_increase)) +
  geom_col()+
  xlab("")+
  ylab("Share of Respondents")+
  theme_bw()+
  ylim(c(0,1))+
  ggtitle("a) Increase environmental protection")+
  theme_demos

# Issue 2: Consumer
eudata$issue2_consumer_rec=recode(eudata$issue2_consumer,
                                  "Increase consumer protection at the cost of more bureaucracy for businesses" = "Increase consumer protection",
                                  "Decrease consumer protection with the benefit of less bureaucracy for businesses" = "Decrease consumer protection")

issue_2_results <- data.frame(table(eudata$issue2_consumer_rec,eudata$country_code))

issue_2_data <- data.frame(country_code=unique(eudata$country_code))
issue_2_data$country_code <- as.character(issue_2_data$country_code)
issue_2_data$share_increase <- NA
for(i in 1:nrow(issue_2_data)){
  country <- issue_2_data$country_code[i]
  issue_2_data$share_increase[issue_2_data$country_code == country] <- issue_2_results$Freq[issue_2_results$Var1 == "Increase consumer protection" & issue_2_results$Var2==country]/
    (issue_2_results$Freq[issue_2_results$Var1 == "Increase consumer protection" & issue_2_results$Var2==country]+
       issue_2_results$Freq[issue_2_results$Var1 == "Decrease consumer protection" & issue_2_results$Var2==country])
}

issue_2_data$share_increase <- round(issue_2_data$share_increase,2)
issue_2_data$country_name <- recode(issue_2_data$country_code,
                                    "ES" = "Spain",
                                    "IT" = "Italy",
                                    "DE" = "Germany",
                                    "FR" = "France",
                                    "PL" = "Poland")

plot_issue_2=ggplot(issue_2_data, aes(x = country_name, y=share_increase)) +
  geom_col()+
  xlab("")+
  ylab("Share of Respondents")+
  theme_bw()+
  ylim(c(0,1))+
  ggtitle("b) Increase consumer protection")+
  theme_demos

# Issue 3: Financial Support
eudata$issue3_finance_rec=recode(eudata$issue3_finance,
                                 "Increase financial support for weak economies, creating costs for economically stronger member states" = "Increase financial support",
                                 "Decrease financial support for weak economies, reducing costs for economically stronger member states" = "Decrease financial support")

issue_3_results <- data.frame(table(eudata$issue3_finance_rec,eudata$country_code))

issue_3_data <- data.frame(country_code=unique(eudata$country_code))
issue_3_data$country_code <- as.character(issue_3_data$country_code)
issue_3_data$share_increase <- NA
for(i in 1:nrow(issue_3_data)){
  country <- issue_3_data$country_code[i]
  issue_3_data$share_increase[issue_3_data$country_code == country] <- issue_3_results$Freq[issue_3_results$Var1 == "Increase financial support" & issue_3_results$Var2==country]/
    (issue_3_results$Freq[issue_3_results$Var1 == "Increase financial support" & issue_3_results$Var2==country]+
       issue_3_results$Freq[issue_3_results$Var1 == "Decrease financial support" & issue_3_results$Var2==country])
}

issue_3_data$share_increase <- round(issue_3_data$share_increase,2)
issue_3_data$country_name <- recode(issue_3_data$country_code,
                                    "ES" = "Spain",
                                    "IT" = "Italy",
                                    "DE" = "Germany",
                                    "FR" = "France",
                                    "PL" = "Poland")

plot_issue_3=ggplot(issue_3_data, aes(x = country_name, y=share_increase)) +
  geom_col()+
  xlab("")+
  ylab("Share of Respondents")+
  theme_bw()+
  ylim(c(0,1))+
  ggtitle("c) Increase financial support")+
  theme_demos

# Issue 4: Military
eudata$issue4_military_rec=recode(eudata$issue4_military,
                                  "Increase military cooperation between member states" = "Increase military cooperation",
                                  "Decrease military cooperation between member states" = "Decrease military cooperation")

issue_4_results <- data.frame(table(eudata$issue4_military_rec,eudata$country_code))

issue_4_data <- data.frame(country_code=unique(eudata$country_code))
issue_4_data$country_code <- as.character(issue_4_data$country_code)
issue_4_data$share_increase <- NA
for(i in 1:nrow(issue_4_data)){
  country <- issue_4_data$country_code[i]
  issue_4_data$share_increase[issue_4_data$country_code == country] <- issue_4_results$Freq[issue_4_results$Var1 == "Increase military cooperation" & issue_4_results$Var2==country]/
    (issue_4_results$Freq[issue_4_results$Var1 == "Increase military cooperation" & issue_4_results$Var2==country]+
       issue_4_results$Freq[issue_4_results$Var1 == "Decrease military cooperation" & issue_4_results$Var2==country])
}

issue_4_data$share_increase <- round(issue_4_data$share_increase,2)
issue_4_data$country_name <- recode(issue_4_data$country_code,
                                    "ES" = "Spain",
                                    "IT" = "Italy",
                                    "DE" = "Germany",
                                    "FR" = "France",
                                    "PL" = "Poland")

plot_issue_4=ggplot(issue_4_data, aes(x = country_name, y=share_increase)) +
  geom_col()+
  xlab("")+
  ylab("Share of Respondents")+
  theme_bw()+
  ylim(c(0,1))+
  ggtitle("d) Increase military cooperation")+
  theme_demos

# Issue 5: Refugees
eudata$issue5_refugees_rec=recode(eudata$issue5_refugees,
                                  "Increase the reallocation of refugees between member states" = "Increase refugee reallocation",
                                  "Decrease the reallocation of refugees between member states" = "Decrease refugee reallocation")

issue_5_results <- data.frame(table(eudata$issue5_refugees_rec,eudata$country_code))

issue_5_data <- data.frame(country_code=unique(eudata$country_code))
issue_5_data$country_code <- as.character(issue_5_data$country_code)
issue_5_data$share_increase <- NA
for(i in 1:nrow(issue_5_data)){
  country <- issue_5_data$country_code[i]
  issue_5_data$share_increase[issue_5_data$country_code == country] <- issue_5_results$Freq[issue_5_results$Var1 == "Increase refugee reallocation" & issue_5_results$Var2==country]/
    (issue_5_results$Freq[issue_5_results$Var1 == "Increase refugee reallocation" & issue_5_results$Var2==country]+
       issue_5_results$Freq[issue_5_results$Var1 == "Decrease refugee reallocation" & issue_5_results$Var2==country])
}

issue_5_data$share_increase <- round(issue_5_data$share_increase,2)
issue_5_data$country_name <- recode(issue_5_data$country_code,
                                    "ES" = "Spain",
                                    "IT" = "Italy",
                                    "DE" = "Germany",
                                    "FR" = "France",
                                    "PL" = "Poland")

plot_issue_5=ggplot(issue_5_data, aes(x = country_name, y=share_increase)) +
  geom_col()+
  xlab("")+
  ylab("Share of Respondents")+
  theme_bw()+
  ylim(c(0,1))+
  ggtitle("e) Increase refugee reallocation")+
  theme_demos

# This is Figure E1
gridExtra::grid.arrange(plot_issue_1,plot_issue_2,plot_issue_3,
                        plot_issue_4,plot_issue_5,ncol=2)

#### Appendix F ####
#### Figure F1 ####

eudata_long$vignette <- as.factor(eudata_long$vignette)
mm_by_vignette=cj(eudata_long, legitimate ~ 
                    consult + 
                    council+
                    ep + 
                    eu_opinion +
                    national_opinion +
                    congruence, id = ~UUID, by = ~vignette, estimate = "mm")

names(mm_by_vignette)[names(mm_by_vignette)=="level"]="term"
names(mm_by_vignette)[names(mm_by_vignette)=="vignette"]="model"

mm_by_vignette$feature <- mm_by_vignette$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_vignette$feature <- factor(mm_by_vignette$feature, levels = c("Egotropic representation",
                                                                    "Sociotropic representation (EU-wide)",
                                                                    "Sociotropic representation (national)",
                                                                    "Consultation",
                                                                    "Voting in Council",
                                                                    "Voting in EP"))

# This is Figure F1
dwplot(mm_by_vignette,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

#### Appendix G ####
#### Figure G1 ####

# Extract duration of screens 5 to 8, which were the vignette screens for the EU
euvignettedurations=data.frame(duration=c(eudata$X.duration...screen.5.,
                                          eudata$X.duration...screen.6.,
                                          eudata$X.duration...screen.7.,
                                          eudata$X.duration...screen.8.,
                                          eudata$X.duration...screen.9.))
# Label vignettes
euvignettedurations$vignette_number=as.factor(c(rep(1,nrow(eudata)),
                                                rep(2,nrow(eudata)),
                                                rep(3,nrow(eudata)),
                                                rep(4,nrow(eudata)),
                                                rep(5,nrow(eudata))))

# Calculate median impression for each vignette number
aggregate(euvignettedurations$duration,by=list(euvignettedurations$vignette_number),median)

# Extract statistics for each vignette to create custom boxplot
stats_1 <- boxplot.stats(euvignettedurations$duration[euvignettedurations$vignette_number==1])$stats
df_1 <- data.frame(x="Vignette 1", ymin=stats_1[1], lower=stats_1[2], middle=stats_1[3], 
                   upper=stats_1[4], ymax=stats_1[5])
stats_2 <- boxplot.stats(euvignettedurations$duration[euvignettedurations$vignette_number==2])$stats
df_2 <- data.frame(x="Vignette 2", ymin=stats_2[1], lower=stats_2[2], middle=stats_2[3], 
                   upper=stats_2[4], ymax=stats_2[5])
stats_3 <- boxplot.stats(euvignettedurations$duration[euvignettedurations$vignette_number==3])$stats
df_3 <- data.frame(x="Vignette 3", ymin=stats_3[1], lower=stats_3[2], middle=stats_3[3], 
                   upper=stats_3[4], ymax=stats_3[5])
stats_4 <- boxplot.stats(euvignettedurations$duration[euvignettedurations$vignette_number==4])$stats
df_4 <- data.frame(x="Vignette 4", ymin=stats_4[1], lower=stats_4[2], middle=stats_4[3], 
                   upper=stats_4[4], ymax=stats_4[5])
stats_5 <- boxplot.stats(euvignettedurations$duration[euvignettedurations$vignette_number==5])$stats
df_5 <- data.frame(x="Vignette 5", ymin=stats_5[1], lower=stats_5[2], middle=stats_5[3], 
                   upper=stats_5[4], ymax=stats_5[5])

# Combine vignette statistics for plot
df=rbind(df_1,df_2,df_3,df_4,df_5)

# This is Figure G1
ggplot(df, aes(x=x, lower=lower, upper=upper, middle=middle, ymin=ymin, 
               ymax=ymax)) + 
  geom_boxplot(stat="identity")+
  xlab("")+
  ylab("Duration in Seconds")+
  theme_bw()+
  theme(panel.background = element_rect(fill = NA),
        panel.grid.minor=element_blank(),
        panel.grid.major = element_blank(),
        axis.text=element_text(colour="black"),
        legend.position="none")

#### Figure G2 ####
euvignettedurations=data.frame(duration=c(eudata$X.duration...screen.5.,
                                          eudata$X.duration...screen.6.,
                                          eudata$X.duration...screen.7.,
                                          eudata$X.duration...screen.8.,
                                          eudata$X.duration...screen.9.))

euvignettedurations$vignette_number=as.factor(c(rep(1,nrow(eudata)),
                                                rep(2,nrow(eudata)),
                                                rep(3,nrow(eudata)),
                                                rep(4,nrow(eudata)),
                                                rep(5,nrow(eudata))))

eudata_long=cbind(eudata_long,euvignettedurations)
eudata_long= eudata_long %>%
  mutate(
    duration_rec = case_when(
      duration <= 12                   ~ "First quartile",
      duration > 12 & duration <= 19   ~ "Second quartile",
      duration > 19 & duration <= 32   ~ "Third quartile",
      duration > 32                    ~ "Fourth quartile"
    )
  )

eudata_long$duration_rec_fac <- factor(eudata_long$duration_rec,levels = c("First quartile",
                                                                           "Second quartile",
                                                                           "Third quartile",
                                                                           "Fourth quartile"))
mm_by_duration=cj(eudata_long, legitimate ~ 
                    consult + 
                    council+
                    ep + 
                    eu_opinion +
                    national_opinion +
                    congruence, id = ~UUID, by = ~duration_rec_fac, estimate = "mm")

names(mm_by_duration)[names(mm_by_duration)=="level"]="term"
names(mm_by_duration)[names(mm_by_duration)=="duration_rec_fac"]="model"

mm_by_duration$feature <- mm_by_duration$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_duration$feature <- factor(mm_by_duration$feature, levels = c("Egotropic representation",
                                                                    "Sociotropic representation (EU-wide)",
                                                                    "Sociotropic representation (national)",
                                                                    "Consultation",
                                                                    "Voting in Council",
                                                                    "Voting in EP"))

mm_by_duration$model <- factor(mm_by_duration$model,levels = c("First quartile",
                                                               "Second quartile",
                                                               "Third quartile",
                                                               "Fourth quartile"))

# This is Figure G2
dwplot(mm_by_duration,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("") +
  theme_results +
  scale_colour_grey(start = 0.25, end = .6,
                    name = "",
                    labels = c("First quartile",
                               "Second quartile",
                               "Third quartile",
                               "Fourth quartile")) +
  scale_shape_discrete(name = "",
                       labels = c("First quartile",
                                  "Second quartile",
                                  "Third quartile",
                                  "Fourth quartile"))

#### Figure G3 ####

eudata_no_satisficing=eudata_long %>% filter(duration_rec!="First quartile")

results_eu_main_no_satisficing <- cregg::mm(legitimate ~ 
                                              consult + 
                                              council+
                                              ep + 
                                              eu_opinion +
                                              national_opinion +
                                              congruence, data = eudata_no_satisficing, id = ~UUID)

names(results_eu_main_no_satisficing)[4]="term"

results_eu_main_no_satisficing$feature <- results_eu_main_no_satisficing$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

results_eu_main_no_satisficing$feature <- factor(results_eu_main_no_satisficing$feature, levels = c("Egotropic representation",
                                                                                                    "Sociotropic representation (EU-wide)",
                                                                                                    "Sociotropic representation (national)",
                                                                                                    "Consultation",
                                                                                                    "Voting in Council",
                                                                                                    "Voting in EP"))

# This is Figure G3
dwplot(results_eu_main_no_satisficing,dot_args = list(size = 2)) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  scale_color_manual(values = c("black")) +
  ggtitle("") + 
  theme_results +
  theme(legend.position = "none")

#### Appendix H ####
#### Figure H1 ####


results_eu_main <- cregg::mm(legitimate ~ 
                               congruence+
                               consult + 
                               council+
                               ep + 
                               eu_opinion +
                               national_opinion, data = eudata_long, id = ~UUID)

names(results_eu_main)[4]="term"

results_eu_main$feature <- results_eu_main$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

results_eu_main$feature <- factor(results_eu_main$feature, levels = c("Egotropic representation",
                                                                      "Sociotropic representation (EU-wide)",
                                                                      "Sociotropic representation (national)",
                                                                      "Consultation",
                                                                      "Voting in Council",
                                                                      "Voting in EP"))

results_eu_main$lower_84 <- results_eu_main$estimate-1.44*results_eu_main$std.error
results_eu_main$upper_84 <- results_eu_main$estimate+1.44*results_eu_main$std.error

results_eu_main$term <- results_eu_main$term %>% 
  recode("<blank>" = "No",
         "In this instance, the European Commission asked EU citizens and interest groups to submit their views on the topic." = "Yes",
         "All national governments voted in favor." = "Unanimity",
         "Most national governments voted in favor including the [COUNTRY] government, some voted against the decision." = "Majority incl. government",
         "Most national governments voted in favor, some voted against the decision including the [COUNTRY] government." = "Majority not incl. government",
         "Most" = "Large majority",
         "Slightly more than half" = "Slight majority", 
         "Most EU citizens oppose this decision" = "No",
         "Most EU citizens support this decision" = "Yes", 
         " most [NATIONALITY] support it" = "Yes", 
         " most [NATIONALITY] oppose it" = "No")

results_eu_main_nocongruence = results_eu_main %>% filter(feature != "Egotropic representation")

# This is Figure H1
results_eu_main_nocongruence %>% 
  ggplot(aes(xmin=lower,xmax=upper,y=term,x=estimate))+
  geom_point(size=3)+
  geom_errorbarh(height=0,size=0.75)+
  geom_errorbarh(height=0,size=2.25,data=results_eu_main_nocongruence,aes(xmin=lower_84,xmax=upper_84,y=term))+
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("") + 
  scale_color_manual(values = c("black")) +
  theme_results

#### Appendix I ####
#### Figure I1 ####
eudata_long <- eudata_long %>% 
  mutate(EU_membership_rec = case_when(
    EU_membership == "A good thing" ~ "Europhiles",
    EU_membership == "A bad thing" ~ "Eurosceptics",
    TRUE ~ "Neither"
  ))
eudata_member <- eudata_long%>%
  filter(EU_membership%in%c("A good thing","A bad thing"))
eudata_member$EU_membership_rec <- as.factor(eudata_member$EU_membership_rec)
eudata_member$EU_membership_rec <- droplevels(eudata_member$EU_membership_rec)

mm_by_eurosceptics <- cj(legitimate ~ 
                           consult + 
                           council+
                           ep + 
                           eu_opinion +
                           national_opinion +
                           congruence, data = eudata_member, id = ~UUID, estimate = "mm", by = ~EU_membership_rec)

names(mm_by_eurosceptics)[names(mm_by_eurosceptics)=="level"]="term"
names(mm_by_eurosceptics)[names(mm_by_eurosceptics)=="EU_membership_rec"]="model"

mm_by_eurosceptics$feature <- mm_by_eurosceptics$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_eurosceptics$feature <- factor(mm_by_eurosceptics$feature, levels = c("Egotropic representation",
                                                                            "Sociotropic representation (EU-wide)",
                                                                            "Sociotropic representation (national)",
                                                                            "Consultation",
                                                                            "Voting in Council",
                                                                            "Voting in EP"))

# Figure I1
dwplot(mm_by_eurosceptics,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

#### Figure I2 ####
eudata_long <- eudata_long %>% 
  mutate(Eurosceptic_voter = case_when(
    vote_pl %in% c("Kukiz'15 (K'15)","Prawo i Sprawiedliwość (PiS)","Wolność") ~ "Eurosceptic Party Supporter",
    vote_fr %in% c("La France Insoumise (FI)","Front National (FN)")           ~ "Eurosceptic Party Supporter",
    vote_es %in% c("Unidos Podemos")                                           ~ "Eurosceptic Party Supporter",
    vote_it %in% c("Fratelli d'Italia (FdI)","Lega (LN)","Movimento 5 Stelle (M5S)") ~ "Eurosceptic Party Supporter",
    vote_de %in% c("Die Linke","Alternative für Deutschland (AfD)") ~ "Eurosceptic Party Supporter",
    TRUE     ~ "Non-Eurosceptic Party Supporter"
  ))
eudata_long$Eurosceptic_voter <- as.factor(eudata_long$Eurosceptic_voter)
mm_by_Eurosceptic_voter <- cj(legitimate ~ 
                                consult + 
                                council+
                                ep + 
                                eu_opinion +
                                national_opinion +
                                congruence, data = eudata_long, id = ~UUID, 
                              estimate = "mm", by = ~Eurosceptic_voter)

names(mm_by_Eurosceptic_voter)[names(mm_by_Eurosceptic_voter)=="level"]="term"
names(mm_by_Eurosceptic_voter)[names(mm_by_Eurosceptic_voter)=="Eurosceptic_voter"]="model"

mm_by_Eurosceptic_voter$feature <- mm_by_Eurosceptic_voter$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_Eurosceptic_voter$feature <- factor(mm_by_Eurosceptic_voter$feature, levels = c("Egotropic representation",
                                                                                      "Sociotropic representation (EU-wide)",
                                                                                      "Sociotropic representation (national)",
                                                                                      "Consultation",
                                                                                      "Voting in Council",
                                                                                      "Voting in EP"))

# Figure I2
dwplot(mm_by_Eurosceptic_voter,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme(plot.title = element_text(face="bold"),
        axis.text=element_text(colour="black"),
        axis.title=element_text(size=8,colour="black"),
        strip.text.y = element_text(angle = 0, face="bold"),
        #          strip.background =element_rect(fill="white"),
        strip.background = element_blank(),
        panel.border = element_blank(), 
        axis.ticks = element_blank(),
        panel.background = element_blank(), 
        #panel.grid = element_blank(),          
        panel.spacing.x = unit(0,"line"),
        legend.position = "bottom",
        legend.title = element_blank())

#### Figure I3 ####

eudata_long <- eudata_long %>% 
  mutate(Populist_voter = case_when(
    vote_pl %in% c("Kukiz'15 (K'15)","Prawo i Sprawiedliwość (PiS)","Wolność") ~ "Populist Party Supporter",
    vote_fr %in% c("La France Insoumise (FI)","Front National (FN)")           ~ "Populist Party Supporter",
    vote_es %in% c("Unidos Podemos")                                           ~ "Populist Party Supporter",
    vote_it %in% c("Lega (LN)","Movimento 5 Stelle (M5S)") ~ "Populist Party Supporter",
    vote_de %in% c("Die Linke","Alternative für Deutschland (AfD)") ~ "Populist Party Supporter",
    TRUE     ~ "Non-Populist Party Supporter"
  ))

eudata_long$Populist_voter <- as.factor(eudata_long$Populist_voter)
mm_by_Populist_voter <- cj(legitimate ~ 
                             consult + 
                             council+
                             ep + 
                             eu_opinion +
                             national_opinion +
                             congruence, data = eudata_long, id = ~UUID, 
                           estimate = "mm", by = ~Populist_voter)

names(mm_by_Populist_voter)[names(mm_by_Populist_voter)=="level"]="term"
names(mm_by_Populist_voter)[names(mm_by_Populist_voter)=="Populist_voter"]="model"

mm_by_Populist_voter$feature <- mm_by_Populist_voter$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_Populist_voter$feature <- factor(mm_by_Populist_voter$feature, levels = c("Egotropic representation",
                                                                                "Sociotropic representation (EU-wide)",
                                                                                "Sociotropic representation (national)",
                                                                                "Consultation",
                                                                                "Voting in Council",
                                                                                "Voting in EP"))

# This is Figure I3
dwplot(mm_by_Populist_voter,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

#### Figure I4 ####

eudata_long$education_rec=eudata_long$education_level
eudata_long$education_rec=recode(eudata_long$education_rec,
                                 "high" = "High",
                                 "medium" = "Medium",
                                 "low" = "Low or None",
                                 "no" = "Low or None")
eudata_long$education_rec=as.factor(eudata_long$education_rec)
eudata_long$education_rec=relevel(eudata_long$education_rec,"Medium")
eudata_long$education_rec=relevel(eudata_long$education_rec,"Low or None")


mm_by_education <- cj(legitimate ~ 
                        consult + 
                        council+
                        ep + 
                        eu_opinion +
                        national_opinion +
                        congruence, data = eudata_long, id = ~UUID, estimate = "mm", by = ~education_rec)
names(mm_by_education)[names(mm_by_education)=="level"]="term"
names(mm_by_education)[names(mm_by_education)=="education_rec"]="model"

mm_by_education$feature <- mm_by_education$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_education$feature <- factor(mm_by_education$feature, levels = c("Egotropic representation",
                                                                      "Sociotropic representation (EU-wide)",
                                                                      "Sociotropic representation (national)",
                                                                      "Consultation",
                                                                      "Voting in Council",
                                                                      "Voting in EP"))

# This is Figure I4
dwplot(mm_by_education,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

#### Figure I5 ####

eudata_long$country_labels=as.factor(eudata_long$country_code)
eudata_long$country_labels=recode(eudata_long$country_labels,
                                  "IT" = "Italy",
                                  "ES" = "Spain",
                                  "DE" = "Germany",
                                  "FR" = "France",
                                  "PL" = "Poland")
mm_by_countries <- cj(legitimate ~ 
                        consult + 
                        council+
                        ep + 
                        eu_opinion +
                        national_opinion +
                        congruence, data = eudata_long, id = ~UUID, estimate = "mm", by = ~country_labels)
names(mm_by_countries)[names(mm_by_countries)=="level"]="term"
names(mm_by_countries)[names(mm_by_countries)=="country_labels"]="model"

mm_by_countries$feature <- mm_by_countries$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_countries$feature <- factor(mm_by_countries$feature, levels = c("Egotropic representation",
                                                                      "Sociotropic representation (EU-wide)",
                                                                      "Sociotropic representation (national)",
                                                                      "Consultation",
                                                                      "Voting in Council",
                                                                      "Voting in EP"))

# Figure I5
dwplot(mm_by_countries,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

#### Figure I6 ####

eudata_long$topic_rec=recode(eudata_long$topic,
                             "increase environmental protection at the cost of more regulation for businesses" = "more environment",
                             "increase the reallocation of refugees between member states" = "more reallocation of refugees",
                             "decrease environmental protection with the benefit of less regulation for businesses" = "less environment",
                             "increase consumer protection at the cost of more bureaucracy for businesses" = "more consumer",
                             "decrease consumer protection with the benefit of less bureaucracy for businesses" = "less consumer",
                             "increase military cooperation between member states" = "more military",
                             "decrease financial support for weak economies, reducing costs for economically stronger member states" = "less financial support",
                             "decrease military cooperation between member states" = "less military",
                             "increase financial support for weak economies, creating costs for economically stronger member states" = "more financial support",
                             "decrease the reallocation of refugees between member states" = "less reallocation of refugees")

eudata_long <- eudata_long %>%
  mutate(
    topic_type = case_when(
      grepl("environment",topic_rec) ~ "environment",
      grepl("consumer",topic_rec) ~ "consumer",
      grepl("financial",topic_rec) ~ "financial",
      grepl("military",topic_rec) ~ "military",
      grepl("refugees",topic_rec) ~ "refugees",
      TRUE ~"other"
    )
  )

eudata_long$topic_type <- as.factor(eudata_long$topic_type)
mm_by_topics <- cj(legitimate ~ 
                     consult + 
                     council+
                     ep + 
                     eu_opinion +
                     national_opinion +
                     congruence, data = eudata_long, id = ~UUID, estimate = "mm", by = ~topic_type)
names(mm_by_topics)[names(mm_by_topics)=="level"]="term"
names(mm_by_topics)[names(mm_by_topics)=="topic_type"]="model"

mm_by_topics$feature <- mm_by_topics$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_topics$feature <- factor(mm_by_topics$feature, levels = c("Egotropic representation",
                                                                      "Sociotropic representation (EU-wide)",
                                                                      "Sociotropic representation (national)",
                                                                      "Consultation",
                                                                      "Voting in Council",
                                                                      "Voting in EP"))

# This is Figure I6
dwplot(mm_by_topics,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

#### Figure I7 ####

mm_by_topics <- cj(legitimate ~ 
                     consult + 
                     council+
                     ep + 
                     opinion +
                     congruence, data = eudata_long, id = ~UUID, estimate = "mm", by = ~topic_type)
names(mm_by_topics)[names(mm_by_topics)=="level"]="term"
names(mm_by_topics)[names(mm_by_topics)=="topic_type"]="model"

mm_by_topics$feature <- mm_by_topics$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "opinion" = "Sociotropic representation")

mm_by_topics$feature <- factor(mm_by_topics$feature, levels = c("Egotropic representation",
                                                                "Sociotropic representation",
                                                                "Consultation",
                                                                "Voting in Council",
                                                                "Voting in EP"))

mm_by_topics_opinion_only = mm_by_topics %>% filter(grepl("Sociotropic",feature)) %>% 
  mutate(model_rec = case_when(
    model=="consumer" ~ "Consumer \nProtection",
    model=="environment" ~ "Environmental \nProtection",
    model=="financial" ~ "Financial \nSupport",
    model=="military" ~ "Military \nCooperation",
    model=="refugees" ~ "Reallocation of \nRefugees"
  ))

# This is Figure I7
dwplot(mm_by_topics_opinion_only,dot_args = list(size = 2,color="black"),whisker_args = list(aes(colour="black"))) %>%
  relabel_predictors(label_changes_topics) +
  facet_grid(model_rec~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  scale_color_manual(values = c("black")) +
  theme(plot.title = element_text(face="bold"),
        axis.text=element_text(colour="black"),
        axis.title=element_text(colour="black"),
        strip.text.y = element_text(angle = 0, face="bold"),
        #          strip.background =element_rect(fill="white"),
        strip.background = element_blank(),
        panel.border = element_blank(), 
        axis.ticks = element_blank(),
        panel.background = element_blank(), 
        #panel.grid = element_blank(),          
        panel.spacing.x = unit(0,"line"),
        legend.position = "none",
        legend.title = element_blank())

#### Figure I8 ####

eudata_long$eu_opinion_rec=recode(eudata_long$eu_opinion,
                                  "Most EU citizens oppose this decision" = "EU against",
                                  "Most EU citizens support this decision" = "EU in favour")

eudata_long$national_opinion_rec=recode(eudata_long$national_opinion,
                                        " most [NATIONALITY] oppose it" = "national against",
                                        " most [NATIONALITY] support it" = "national in favour")

eudata_long$public_personal_congruence=paste(eudata_long$eu_opinion_rec,eudata_long$national_opinion_rec,eudata_long$congruence)

mod_duration=lm(duration~public_personal_congruence,data=eudata_long%>%
                  filter(duration>=quantile(eudata_long$duration,0.05)&
                           duration<=quantile(eudata_long$duration,0.95)))


nd <- with(eudata_long,
           expand.grid(public_personal_congruence = unique(public_personal_congruence)))

pred_list <- predict(mod_duration, nd, se.fit = TRUE)
preds <- with(pred_list,
              data.frame(nd, prob = fit,
                         lower = fit - 1.96 * se.fit,
                         upper = fit + 1.96 * se.fit,stringsAsFactors = F))

preds$term=recode(preds$public_personal_congruence,
                  "EU against national against No" = "EU - ; National - ; Self -",
                  "EU against national against Yes" = "EU - ; National - ; Self +",
                  "EU against national in favour No" = "EU - ; National + ; Self -",
                  "EU against national in favour Yes" = "EU - ; National + ; Self +",
                  "EU in favour national against No" = "EU + ; National - ; Self -",
                  "EU in favour national against Yes" = "EU + ; National - ; Self +",
                  "EU in favour national in favour Yes" = "EU + ; National + ; Self +",
                  "EU in favour national in favour No" = "EU + ; National + ; Self -")

#  This is Figure I8
ggplot(data=preds) +
  ggtitle("")+
  geom_pointrange(
    aes(x=reorder(term,prob), y=prob, ymin=lower, ymax=upper),
    lwd = 1/2, position = position_dodge(width = 1/2),
    shape = 21, fill = "black") +
  coord_flip() +
  xlab("") +
  ylab("Response Latencies (in Seconds)") +
  theme_bw()+
  theme(panel.grid.major=element_blank(),
        panel.grid.minor.y = element_blank(),
        panel.grid.minor.x = element_line(),
        axis.text=element_text(colour="black"),
        axis.title=element_text(size=8,colour="black"),
        legend.background = element_rect(color = "black",
                                         fill = "white", size = 0.5, linetype = "solid"),
        #       legend.text=element_text(size=14),
        #legend.title=element_text(size=16),
        legend.position="none")

#### Figure I9 ####
eudata_long = eudata_long %>%
  mutate(
    topic_impression_rec_median = case_when(
      topic_impression <= 52                           ~ "Strong Issue Attitudes",
      topic_impression > 52   ~ "Weak Issue Attitudes"
    )
  )
eudata_long$topic_impression_rec_median <- as.factor(eudata_long$topic_impression_rec_median)
mm_by_topicimpression=cj(eudata_long%>%filter(duration>8), legitimate ~ 
                           consult + 
                           council+
                           ep + 
                           eu_opinion +
                           national_opinion +
                           congruence, id = ~UUID, by = ~topic_impression_rec_median, estimate = "mm")

names(mm_by_topicimpression)[names(mm_by_topicimpression)=="level"]="term"
names(mm_by_topicimpression)[names(mm_by_topicimpression)=="topic_impression_rec_median"]="model"

mm_by_topicimpression$feature <- mm_by_topicimpression$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_topicimpression$feature <- factor(mm_by_topicimpression$feature, levels = c("Egotropic representation",
                                                                                  "Sociotropic representation (EU-wide)",
                                                                                  "Sociotropic representation (national)",
                                                                                  "Consultation",
                                                                                  "Voting in Council",
                                                                                  "Voting in EP"))

mm_by_topicimpression_opinion_only = mm_by_topicimpression %>% filter(grepl("Sociotropic",feature))

# Figure I9
dwplot(mm_by_topicimpression_opinion_only,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

#### Figure I10 ####

mm_by_topicimpression_de=cj(eudata_long%>%filter(duration>8&country_code=="DE"), legitimate ~ 
                              consult + 
                              council+
                              ep + 
                              eu_opinion +
                              national_opinion +
                              congruence, id = ~UUID, by = ~topic_impression_rec_median, estimate = "amce")
mm_by_topicimpression_de <- mm_by_topicimpression_de %>% filter(level%in%c("Most EU citizens oppose this decision",
                                                                           "Most EU citizens support this decision",
                                                                           " most [NATIONALITY] oppose it",
                                                                           " most [NATIONALITY] support it"))
names(mm_by_topicimpression_de)[names(mm_by_topicimpression_de)=="level"]="term"
names(mm_by_topicimpression_de)[names(mm_by_topicimpression_de)=="topic_impression_rec_median"]="model"
mm_by_topicimpression_de$feature <- mm_by_topicimpression_de$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")
mm_by_topicimpression_de$feature <- factor(mm_by_topicimpression_de$feature, levels = c("Egotropic representation",
                                                                                        "Sociotropic representation (EU-wide)",
                                                                                        "Sociotropic representation (national)",
                                                                                        "Consultation",
                                                                                        "Voting in Council",
                                                                                        "Voting in EP"))

mech_1_de <- dwplot(mm_by_topicimpression_de,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  geom_vline(xintercept=0)+
  ggtitle("b) Germany") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

mm_by_topicimpression_it=cj(eudata_long%>%filter(duration>8&country_code=="IT"), legitimate ~ 
                              consult + 
                              council+
                              ep + 
                              eu_opinion +
                              national_opinion +
                              congruence, id = ~UUID, by = ~topic_impression_rec_median, estimate = "amce")
mm_by_topicimpression_it <- mm_by_topicimpression_it %>% filter(level%in%c("Most EU citizens oppose this decision",
                                                                           "Most EU citizens support this decision",
                                                                           " most [NATIONALITY] oppose it",
                                                                           " most [NATIONALITY] support it"))
names(mm_by_topicimpression_it)[names(mm_by_topicimpression_it)=="level"]="term"
names(mm_by_topicimpression_it)[names(mm_by_topicimpression_it)=="topic_impression_rec_median"]="model"
mm_by_topicimpression_it$feature <- mm_by_topicimpression_it$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")
mm_by_topicimpression_it$feature <- factor(mm_by_topicimpression_it$feature, levels = c("Egotropic representation",
                                                                                        "Sociotropic representation (EU-wide)",
                                                                                        "Sociotropic representation (national)",
                                                                                        "Consultation",
                                                                                        "Voting in Council",
                                                                                        "Voting in EP"))
mech_1_it <- dwplot(mm_by_topicimpression_it,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  geom_vline(xintercept=0)+
  ggtitle("c) Italy") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

mm_by_topicimpression_es=cj(eudata_long%>%filter(duration>8&country_code=="ES"), legitimate ~ 
                              consult + 
                              council+
                              ep + 
                              eu_opinion +
                              national_opinion +
                              congruence, id = ~UUID, by = ~topic_impression_rec_median, estimate = "amce")
mm_by_topicimpression_es <- mm_by_topicimpression_es %>% filter(level%in%c("Most EU citizens oppose this decision",
                                                                           "Most EU citizens support this decision",
                                                                           " most [NATIONALITY] oppose it",
                                                                           " most [NATIONALITY] support it"))
names(mm_by_topicimpression_es)[names(mm_by_topicimpression_es)=="level"]="term"
names(mm_by_topicimpression_es)[names(mm_by_topicimpression_es)=="topic_impression_rec_median"]="model"
mm_by_topicimpression_es$feature <- mm_by_topicimpression_es$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")
mm_by_topicimpression_es$feature <- factor(mm_by_topicimpression_es$feature, levels = c("Egotropic representation",
                                                                                        "Sociotropic representation (EU-wide)",
                                                                                        "Sociotropic representation (national)",
                                                                                        "Consultation",
                                                                                        "Voting in Council",
                                                                                        "Voting in EP"))
mech_1_es <- dwplot(mm_by_topicimpression_es,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  geom_vline(xintercept=0)+
  ggtitle("e) Spain") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

mm_by_topicimpression_fr=cj(eudata_long%>%filter(duration>8&country_code=="FR"), legitimate ~ 
                              consult + 
                              council+
                              ep + 
                              eu_opinion +
                              national_opinion +
                              congruence, id = ~UUID, by = ~topic_impression_rec_median, estimate = "amce")
mm_by_topicimpression_fr <- mm_by_topicimpression_fr %>% filter(level%in%c("Most EU citizens oppose this decision",
                                                                           "Most EU citizens support this decision",
                                                                           " most [NATIONALITY] oppose it",
                                                                           " most [NATIONALITY] support it"))
names(mm_by_topicimpression_fr)[names(mm_by_topicimpression_fr)=="level"]="term"
names(mm_by_topicimpression_fr)[names(mm_by_topicimpression_fr)=="topic_impression_rec_median"]="model"
mm_by_topicimpression_fr$feature <- mm_by_topicimpression_fr$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")
mm_by_topicimpression_fr$feature <- factor(mm_by_topicimpression_fr$feature, levels = c("Egotropic representation",
                                                                                        "Sociotropic representation (EU-wide)",
                                                                                        "Sociotropic representation (national)",
                                                                                        "Consultation",
                                                                                        "Voting in Council",
                                                                                        "Voting in EP"))
mech_1_fr <- dwplot(mm_by_topicimpression_fr,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  geom_vline(xintercept=0)+
  ggtitle("a) France") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results


mm_by_topicimpression_pl=cj(eudata_long%>%filter(duration>8&country_code=="PL"), legitimate ~ 
                              consult + 
                              council+
                              ep + 
                              eu_opinion +
                              national_opinion +
                              congruence, id = ~UUID, by = ~topic_impression_rec_median, estimate = "amce")
mm_by_topicimpression_pl <- mm_by_topicimpression_pl %>% filter(level%in%c("Most EU citizens oppose this decision",
                                                                           "Most EU citizens support this decision",
                                                                           " most [NATIONALITY] oppose it",
                                                                           " most [NATIONALITY] support it"))
names(mm_by_topicimpression_pl)[names(mm_by_topicimpression_pl)=="level"]="term"
names(mm_by_topicimpression_pl)[names(mm_by_topicimpression_pl)=="topic_impression_rec_median"]="model"
mm_by_topicimpression_pl$feature <- mm_by_topicimpression_pl$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")
mm_by_topicimpression_pl$feature <- factor(mm_by_topicimpression_pl$feature, levels = c("Egotropic representation",
                                                                                        "Sociotropic representation (EU-wide)",
                                                                                        "Sociotropic representation (national)",
                                                                                        "Consultation",
                                                                                        "Voting in Council",
                                                                                        "Voting in EP"))
mech_1_pl <- dwplot(mm_by_topicimpression_pl,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  geom_vline(xintercept=0)+
  ggtitle("d) Poland") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

# This is Figure I10
gridExtra::grid.arrange(mech_1_fr,mech_1_de,mech_1_it,mech_1_pl,mech_1_es,ncol=2)

#### Figure I11 ####

eudata_long$agree_with_maj <- 0

for(i in 1:nrow(eudata_long)){
  if(eudata_long$country_code[i]%in%c("DE","ES","IT","FR")){
    if(eudata_long$issue1_environment[i]=="Increase environmental protection at the cost of more regulation for businesses"){
      eudata_long$agree_with_maj[i] <- eudata_long$agree_with_maj[i]+1
    }
    if(eudata_long$issue2_consumer[i]=="Increase consumer protection at the cost of more bureaucracy for businesses"){
      eudata_long$agree_with_maj[i] <- eudata_long$agree_with_maj[i]+1
    }
    if(eudata_long$issue3_finance[i]=="Increase financial support for weak economies, creating costs for economically stronger member states"){
      eudata_long$agree_with_maj[i] <- eudata_long$agree_with_maj[i]+1
    }
    if(eudata_long$issue4_military[i]=="Increase military cooperation between member states"){
      eudata_long$agree_with_maj[i] <- eudata_long$agree_with_maj[i]+1
    }
    if(eudata_long$issue5_refugees[i]=="Increase the reallocation of refugees between member states"){
      eudata_long$agree_with_maj[i] <- eudata_long$agree_with_maj[i]+1
    }
  }
  if(eudata_long$country_code[i]=="PL"){
    if(eudata_long$issue1_environment[i]=="Increase environmental protection at the cost of more regulation for businesses"){
      eudata_long$agree_with_maj[i] <- eudata_long$agree_with_maj[i]+1
    }
    if(eudata_long$issue2_consumer[i]=="Increase consumer protection at the cost of more bureaucracy for businesses"){
      eudata_long$agree_with_maj[i] <- eudata_long$agree_with_maj[i]+1
    }
    if(eudata_long$issue3_finance[i]=="Increase financial support for weak economies, creating costs for economically stronger member states"){
      eudata_long$agree_with_maj[i] <- eudata_long$agree_with_maj[i]+1
    }
    if(eudata_long$issue4_military[i]=="Increase military cooperation between member states"){
      eudata_long$agree_with_maj[i] <- eudata_long$agree_with_maj[i]+1
    }
    if(eudata_long$issue5_refugees[i]=="Decrease the reallocation of refugees between member states"){
      eudata_long$agree_with_maj[i] <- eudata_long$agree_with_maj[i]+1
    }
  }
}

eudata_long <- eudata_long %>% 
  mutate(prospect_alternation = case_when(
    agree_with_maj >3 ~ "Preference Normals",
    TRUE  ~ "Preference Outliers"
  ))
table(eudata_long$agree_with_maj,eudata_long$prospect_alternation)
table(eudata_long$agree_with_maj)

eudata_long$prospect_alternation <- as.factor(eudata_long$prospect_alternation)
mm_by_prospect <- cj(legitimate ~ 
                       consult + 
                       council+
                       ep + 
                       eu_opinion +
                       national_opinion +
                       congruence, data = eudata_long, id = ~UUID, 
                     estimate = "mm", by = ~prospect_alternation)

names(mm_by_prospect)[names(mm_by_prospect)=="level"]="term"
names(mm_by_prospect)[names(mm_by_prospect)=="prospect_alternation"]="model"

mm_by_prospect$feature <- mm_by_prospect$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_prospect$feature <- factor(mm_by_prospect$feature, levels = c("Egotropic representation",
                                                                    "Sociotropic representation (EU-wide)",
                                                                    "Sociotropic representation (national)",
                                                                    "Consultation",
                                                                    "Voting in Council",
                                                                    "Voting in EP"))
mm_by_prospect <- mm_by_prospect %>% filter(grepl("Sociotropic",feature))

# This is Figure I11
dwplot(mm_by_prospect,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

#### Appendix J ####

#### Table J1 ####

results_eu_main_amce <- cregg::amce(legitimate ~ 
                                      consult + 
                                      council+
                                      ep + 
                                      eu_opinion +
                                      national_opinion +
                                      congruence, data = eudata_long, id = ~UUID)
results_eu_main_amce$level=recode(results_eu_main_amce$level,
                                  "<blank>" = "No",
                                  "In this instance, the European Commission asked EU citizens and interest groups to submit their views on the topic." = "Yes",
                                  "All national governments voted in favor." = "Unanimity",
                                  "Most national governments voted in favor including the [COUNTRY] government, some voted against the decision." = "Majority incl. government",
                                  "Most national governments voted in favor, some voted against the decision including the [COUNTRY] government." = "Majority not incl. government",
                                  "Most" = "Large majority",
                                  "Slightly more than half" = "Slight majority",
                                  "Most EU citizens oppose this decision" = "No",
                                  "Most EU citizens support this decision" = "Yes",
                                  " most [NATIONALITY] oppose it" = "No",
                                  " most [NATIONALITY] support it" = "Yes")
results_eu_main_amce$feature=recode(results_eu_main_amce$feature,
                                    "consult" = "Consultation",
                                    "council" = "Voting in Council",
                                    "ep" = "Voting in EP",
                                    "eu_opinion" = "Sociotropic representation (EU-wide)",
                                    "national_opinion" = "Sociotropic representation (national)",
                                    "congruence" = "Egotropic representation")
results_eu_main_amce <- results_eu_main_amce[c(12,13,8,9,10,11,1,2,3,4,5,6,7),]

names(results_eu_main_amce) <- c("Outcome","statistic","Feature","Level","Estimate","Std.Error","z","p","Lower","Upper")

results_eu_main_amce_accept <- cregg::amce(accept ~ 
                                             consult + 
                                             council+
                                             ep + 
                                             eu_opinion +
                                             national_opinion +
                                             congruence, data = eudata_long, id = ~UUID)
results_eu_main_amce_accept$level=recode(results_eu_main_amce_accept$level,
                                         "<blank>" = "No",
                                         "In this instance, the European Commission asked EU citizens and interest groups to submit their views on the topic." = "Yes",
                                         "All national governments voted in favor." = "Unanimity",
                                         "Most national governments voted in favor including the [COUNTRY] government, some voted against the decision." = "Majority incl. government",
                                         "Most national governments voted in favor, some voted against the decision including the [COUNTRY] government." = "Majority not incl. government",
                                         "Most" = "Large majority",
                                         "Slightly more than half" = "Slight majority",
                                         "Most EU citizens oppose this decision" = "No",
                                         "Most EU citizens support this decision" = "Yes",
                                         " most [NATIONALITY] oppose it" = "No",
                                         " most [NATIONALITY] support it" = "Yes")
results_eu_main_amce_accept$feature=recode(results_eu_main_amce_accept$feature,
                                           "consult" = "Consultation",
                                           "council" = "Voting in Council",
                                           "ep" = "Voting in EP",
                                           "eu_opinion" = "Sociotropic representation (EU-wide)",
                                           "national_opinion" = "Sociotropic representation (national)",
                                           "congruence" = "Egotropic representation")
results_eu_main_amce_accept <- results_eu_main_amce_accept[c(12,13,8,9,10,11,1,2,3,4,5,6,7),]

names(results_eu_main_amce_accept) <- c("Outcome","statistic","Feature","Level","Estimate","Std.Error","z","p","Lower","Upper")

# This is Table J1
print(xtable::xtable(rbind(results_eu_main_amce %>% 
                             select(-statistic),
                           results_eu_main_amce_accept %>% 
                             select(-statistic)), file="amce_table.tex"),
      include.rownames=FALSE)

#### Table J2 ####

eudata_long$EU_identity_rec=recode(eudata_long$EU_identity,
                                   "[COUNTRY adj] and European" = "Some European Identity",
                                   "[COUNTRY adj] only" = "Exclusive National Identity",
                                   "European and [COUNTRY adj]" = "Some European Identity",
                                   "European only" = "Some European Identity")

eudata_identity <- eudata_long %>% filter(EU_identity_rec != "None of the above")
eudata_identity$EU_identity_rec <- as.factor(eudata_identity$EU_identity_rec)

results_eu_mech1_amce <- cj(legitimate ~ 
                              consult + 
                              council+
                              ep + 
                              eu_opinion +
                              national_opinion +
                              congruence, data = eudata_identity, 
                            estimate = "amce",id = ~UUID, by = ~EU_identity_rec)

results_eu_mech1_amce$level=recode(results_eu_mech1_amce$level,
                                   "<blank>" = "No",
                                   "In this instance, the European Commission asked EU citizens and interest groups to submit their views on the topic." = "Yes",
                                   "All national governments voted in favor." = "Unanimity",
                                   "Most national governments voted in favor including the [COUNTRY] government, some voted against the decision." = "Majority incl. government",
                                   "Most national governments voted in favor, some voted against the decision including the [COUNTRY] government." = "Majority not incl. government",
                                   "Most" = "Large majority",
                                   "Slightly more than half" = "Slight majority",
                                   "Most EU citizens oppose this decision" = "No",
                                   "Most EU citizens support this decision" = "Yes",
                                   " most [NATIONALITY] oppose it" = "No",
                                   " most [NATIONALITY] support it" = "Yes")
results_eu_mech1_amce$feature=recode(results_eu_mech1_amce$feature,
                                     "consult" = "Consultation",
                                     "council" = "Voting in Council",
                                     "ep" = "Voting in EP",
                                     "eu_opinion" = "Sociotropic representation (EU-wide)",
                                     "national_opinion" = "Sociotropic representation (national)",
                                     "congruence" = "Egotropic representation")
results_eu_mech1_amce <- results_eu_mech1_amce[c(12,13,8,9,10,11,1,2,3,4,5,6,7,25,26,21,22,23,24,14,15,16,17,18,19,20),]


names(results_eu_mech1_amce) <- c("EU Identity","Outcome","statistic","Feature","Level","Estimate","Std.Error","z","p","Lower","Upper","EU_identity_rec")

# This is table J2
print(xtable::xtable(results_eu_mech1_amce %>% 
                       select(-statistic,-Outcome,-EU_identity_rec) %>% 
                       filter(grepl("Sociotropic",Feature)) %>% 
                       arrange(Feature), 
                     file="amce_table_mech1.tex"),
      include.rownames=FALSE)

#### Table J3 ####

eudata_long = eudata_long %>%
  mutate(
    topic_impression_rec = case_when(
      topic_impression <= 38                           ~ "First quartile",
      topic_impression > 38 & topic_impression <= 52   ~ "Second quartile",
      topic_impression > 52 & topic_impression <= 79   ~ "Third quartile",
      topic_impression > 79                            ~ "Fourth quartile"
    )
  )
eudata_long = eudata_long %>%
  mutate(
    topic_impression_rec_median = case_when(
      topic_impression <= 52                           ~ "Strong Issue Attitudes",
      topic_impression > 52   ~ "Weak Issue Attitudes"
    )
  )

eudata_long$topic_impression_rec_median <- as.factor(eudata_long$topic_impression_rec_median)

results_eu_mech2_amce <- cj(eudata_long, legitimate ~ 
                              consult + 
                              council+
                              ep + 
                              eu_opinion +
                              national_opinion +
                              congruence, id = ~UUID, by = ~topic_impression_rec_median, estimate = "amce")

results_eu_mech2_amce$level=recode(results_eu_mech2_amce$level,
                                   "<blank>" = "No",
                                   "In this instance, the European Commission asked EU citizens and interest groups to submit their views on the topic." = "Yes",
                                   "All national governments voted in favor." = "Unanimity",
                                   "Most national governments voted in favor including the [COUNTRY] government, some voted against the decision." = "Majority incl. government",
                                   "Most national governments voted in favor, some voted against the decision including the [COUNTRY] government." = "Majority not incl. government",
                                   "Most" = "Large majority",
                                   "Slightly more than half" = "Slight majority",
                                   "Most EU citizens oppose this decision" = "No",
                                   "Most EU citizens support this decision" = "Yes",
                                   " most [NATIONALITY] oppose it" = "No",
                                   " most [NATIONALITY] support it" = "Yes")
results_eu_mech2_amce$feature=recode(results_eu_mech2_amce$feature,
                                     "consult" = "Consultation",
                                     "council" = "Voting in Council",
                                     "ep" = "Voting in EP",
                                     "eu_opinion" = "Sociotropic representation (EU-wide)",
                                     "national_opinion" = "Sociotropic representation (national)",
                                     "congruence" = "Egotropic representation")
results_eu_mech2_amce <- results_eu_mech2_amce[c(12,13,8,9,10,11,1,2,3,4,5,6,7,25,26,21,22,23,24,14,15,16,17,18,19,20),]

names(results_eu_mech2_amce) <- c("Issue Attitudes","Outcome","statistic","Feature","Level","Estimate","Std.Error","z","p","Lower","Upper","EU_identity_rec")

# This is Table J3
print(xtable::xtable(results_eu_mech2_amce %>% 
                       select(-statistic,-Outcome,-EU_identity_rec) %>% 
                       filter(grepl("Sociotropic",Feature)) %>% 
                       arrange(Feature), 
                     file="amce_table_mech1.tex"),
      include.rownames=FALSE)

#### Table J4 ####

eudata_long <- eudata_long %>% 
  mutate(political_extremes = case_when(
    lire %in% c(0,1,2,8,9,10) ~ "Ideological Extremists",
    TRUE  ~ "Moderates"
  ))

eudata_long$political_extremes <- as.factor(eudata_long$political_extremes)

results_eu_mech3_amce <- cj(legitimate ~ 
                              consult + 
                              council+
                              ep + 
                              eu_opinion +
                              national_opinion +
                              congruence, data = eudata_long, id = ~UUID, 
                            estimate = "amce", by = ~political_extremes)

results_eu_mech3_amce$level=recode(results_eu_mech3_amce$level,
                                   "<blank>" = "No",
                                   "In this instance, the European Commission asked EU citizens and interest groups to submit their views on the topic." = "Yes",
                                   "All national governments voted in favor." = "Unanimity",
                                   "Most national governments voted in favor including the [COUNTRY] government, some voted against the decision." = "Majority incl. government",
                                   "Most national governments voted in favor, some voted against the decision including the [COUNTRY] government." = "Majority not incl. government",
                                   "Most" = "Large majority",
                                   "Slightly more than half" = "Slight majority",
                                   "Most EU citizens oppose this decision" = "No",
                                   "Most EU citizens support this decision" = "Yes",
                                   " most [NATIONALITY] oppose it" = "No",
                                   " most [NATIONALITY] support it" = "Yes")
results_eu_mech3_amce$feature=recode(results_eu_mech3_amce$feature,
                                     "consult" = "Consultation",
                                     "council" = "Voting in Council",
                                     "ep" = "Voting in EP",
                                     "eu_opinion" = "Sociotropic representation (EU-wide)",
                                     "national_opinion" = "Sociotropic representation (national)",
                                     "congruence" = "Egotropic representation")
results_eu_mech3_amce <- results_eu_mech3_amce[c(12,13,8,9,10,11,1,2,3,4,5,6,7,25,26,21,22,23,24,14,15,16,17,18,19,20),]


names(results_eu_mech3_amce) <- c("Ideological Extremes","Outcome","statistic","Feature","Level","Estimate","Std.Error","z","p","Lower","Upper","EU_identity_rec")

# This is Table J4
print(xtable::xtable(results_eu_mech3_amce %>% 
                       select(-statistic,-Outcome,-EU_identity_rec) %>% 
                       filter(grepl("Sociotropic",Feature)) %>% 
                       arrange(Feature), file="amce_table_mech1.tex"),
      include.rownames=FALSE)

#### Table J5 ####

amce_non_congruent <- cj(legitimate ~ 
                           consult + 
                           council+
                           ep + 
                           eu_opinion +
                           national_opinion , data = eudata_long, by = ~congruence, estimate = "amce", id = ~UUID)

amce_non_congruent <- amce_non_congruent %>% 
  filter(feature %in% c("eu_opinion","national_opinion"))

amce_non_congruent$Feature <- amce_non_congruent$Feature %>% 
  recode("eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

amce_non_congruent$Level <- amce_non_congruent$Level %>% 
  recode("Most EU citizens oppose this decision" = "No",
         "Most EU citizens support this decision" = "Yes",
         " most [NATIONALITY] oppose it" = "No",
         " most [NATIONALITY] support it" = "Yes")

names(amce_non_congruent) <- c("Egotropic Representation","Outcome","statistic","Feature","Level","Estimate","Std.Error","z","p","Lower","Upper","EU_identity_rec")

# This is Table J5
print(xtable::xtable(amce_non_congruent %>% select(-statistic,-Outcome,-EU_identity_rec), file="amce_table_congruence.tex"),
      include.rownames=FALSE)

#### Table J6 ####

mod_mech_1 <- lm(legitimate ~ EU_identity_rec*(eu_opinion + national_opinion) + consult + council+ep  +
                   congruence, data = eudata_identity)
mod_mech_2 <- lm(legitimate ~ topic_impression_rec_median*(eu_opinion + national_opinion) + consult + council+ep  +
                   congruence, data = eudata_long)
mod_mech_3 <- lm(legitimate ~ political_extremes*(eu_opinion + national_opinion) + consult + council+ep  +
                   congruence, data = eudata_long)

mod_1_ses <- lmtest::coeftest(mod_mech_1, vcov=plm::vcovHC(mod_mech_1, type="HC1", cluster="UUID")) [,2]
mod_2_ses <- lmtest::coeftest(mod_mech_2, vcov=plm::vcovHC(mod_mech_2, type="HC1", cluster="UUID")) [,2]
mod_3_ses <- lmtest::coeftest(mod_mech_3, vcov=plm::vcovHC(mod_mech_3, type="HC1", cluster="UUID")) [,2]

# This is Table J6
stargazer::stargazer(mod_mech_1,mod_mech_2,mod_mech_3,
                     order = 12,
                     covariate.labels = c("Egotropic Representation",
                                          "Some European Identity",
                                          "Weak Issue Attitudes",
                                          "Preference Normals",
                                          "Sociotropic representation (EU-wide)",
                                          "Sociotropic representation (national)",
                                          "Consultation: Yes",
                                          "Council: Majority incl. government",
                                          "Council: Majority not incl. government",
                                          "EP: Slight majority",
                                          "Mech 1: Some European Identity x Sociotropic representation (EU-wide)",
                                          "Mech 1: Some European Identity x Sociotropic representation (national)",
                                          "Mech 2: Weak Issue Attitudes x Sociotropic representation (EU-wide)",
                                          "Mech 2: Weak Issue Attitudes x Sociotropic representation (national)",
                                          "Mech 3: Moderates x Sociotropic representation (EU-wide)",
                                          "Mech 3: Moderates x Sociotropic representation (national)",
                                          "Intercept"),dep.var.caption = "Rating of Legitimacy of Decision",
                     omit.stat = "F",
                     se = list(mod_1_ses,mod_2_ses,mod_3_ses))


#### Table J7 ####

mod_congruence <- lm(legitimate ~ congruence*(eu_opinion + national_opinion) + consult + council+ep, 
                     data = eudata_long)
mod_congruence_all <- lm(legitimate ~ congruence*(eu_opinion + national_opinion + consult + council + ep), 
                         data = eudata_long)

mod_congruence_ses <- lmtest::coeftest(mod_congruence, vcov=plm::vcovHC(mod_congruence, type="HC1", cluster="UUID")) [,2]
mod_congruence_all_ses <- lmtest::coeftest(mod_congruence_all, vcov=plm::vcovHC(mod_congruence_all, type="HC1", cluster="UUID")) [,2]

# This is Table J7
stargazer::stargazer(mod_congruence,mod_congruence_all,
                     # type="text",
                     covariate.labels = c("Egotropic Representation",
                                          "Sociotropic representation (EU-wide)",
                                          "Sociotropic representation (national)",
                                          "Consultation: Yes",
                                          "Council: Majority incl. government",
                                          "Council: Majority not incl. government",
                                          "EP: Slight majority",
                                          "Egotropic Representation x Sociotropic representation (EU-wide)",
                                          "Egotropic Representation x Sociotropic representation (national)",
                                          "Egotropic Representation x Consultation: Yes",
                                          "Egotropic Representation x Council: Majority incl. government",
                                          "Egotropic Representation x Council: Majority not incl. government",
                                          "Egotropic Representation x EP: Slight majority",
                                          "Intercept"),dep.var.caption = "Rating of Legitimacy of Decision",
                     se = list(mod_congruence_ses,mod_congruence_all_ses))

#### Appendix K ####

#### Figure K1 ####

results_eu_main_accept <- cregg::mm(accept ~ 
                                      consult + 
                                      council+
                                      ep + 
                                      eu_opinion +
                                      national_opinion +
                                      congruence, data = eudata_long, id = ~UUID)

names(results_eu_main_accept)[4]="term"

results_eu_main_accept$feature <- results_eu_main_accept$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

results_eu_main_accept$feature <- factor(results_eu_main_accept$feature, levels = c("Egotropic representation",
                                                                                    "Sociotropic representation (EU-wide)",
                                                                                    "Sociotropic representation (national)",
                                                                                    "Consultation",
                                                                                    "Voting in Council",
                                                                                    "Voting in EP"))

# This is Figure K1
dwplot(results_eu_main_accept,dot_args = list(size = 2)) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Willingness to Accept Decision") + 
  ylab("") +
  scale_color_manual(values = c("black")) +
  ggtitle("") + 
  theme_results + 
  theme(legend.position = "none")

#### Figure K2 ####

mm_non_congruent_accept <- cregg::mm(accept ~ 
                                       consult + 
                                       council+
                                       ep + 
                                       eu_opinion +
                                       national_opinion , data = eudata_long%>%filter(congruence=="No"), id = ~UUID)
names(mm_non_congruent_accept)[4]="term"

mm_non_congruent_accept$feature <- mm_non_congruent_accept$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_non_congruent_accept$feature <- factor(mm_non_congruent_accept$feature, levels = c("Egotropic representation",
                                                                                      "Sociotropic representation (EU-wide)",
                                                                                      "Sociotropic representation (national)",
                                                                                      "Consultation",
                                                                                      "Voting in Council",
                                                                                      "Voting in EP"))

# This is Figure K2
dwplot(mm_non_congruent_accept,dot_args = list(size = 2)) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Willingness to Accept Decision") + 
  ylab("") +
  ggtitle("") + 
  scale_color_manual(values = c("black")) +
  theme_results +
  theme(legend.position = "none")

#### Figure K3 ####

mm_by_identity_accept <- cj(accept ~ 
                              consult + 
                              council+
                              ep + 
                              eu_opinion +
                              national_opinion +
                              congruence, data = eudata_identity, id = ~UUID, 
                            estimate = "mm", by = ~EU_identity_rec)

names(mm_by_identity_accept)[names(mm_by_identity_accept)=="level"]="term"
names(mm_by_identity_accept)[names(mm_by_identity_accept)=="EU_identity_rec"]="model"

mm_by_identity_accept$feature <- mm_by_identity_accept$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_identity_accept$feature <- factor(mm_by_identity_accept$feature, levels = c("Egotropic representation",
                                                                                  "Sociotropic representation (EU-wide)",
                                                                                  "Sociotropic representation (national)",
                                                                                  "Consultation",
                                                                                  "Voting in Council",
                                                                                  "Voting in EP"))

# This is Figure K3
dwplot(mm_by_identity_accept%>% filter(grepl("Sociotropic",feature)),dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Willingness to Accept Decision") + 
  ylab("") +
  ggtitle("") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

#### Figure K4 ####

mm_by_topicimpression_accept=cj(eudata_long%>%filter(duration>8), accept ~ 
                                  consult + 
                                  council+
                                  ep + 
                                  eu_opinion +
                                  national_opinion +
                                  congruence, id = ~UUID, by = ~topic_impression_rec_median, estimate = "mm")

names(mm_by_topicimpression_accept)[names(mm_by_topicimpression_accept)=="level"]="term"
names(mm_by_topicimpression_accept)[names(mm_by_topicimpression_accept)=="topic_impression_rec_median"]="model"

mm_by_topicimpression_accept$feature <- mm_by_topicimpression_accept$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_topicimpression_accept$feature <- factor(mm_by_topicimpression_accept$feature, levels = c("Egotropic representation",
                                                                                                "Sociotropic representation (EU-wide)",
                                                                                                "Sociotropic representation (national)",
                                                                                                "Consultation",
                                                                                                "Voting in Council",
                                                                                                "Voting in EP"))

# Figure K4
dwplot(mm_by_topicimpression_accept %>% filter(grepl("Sociotropic",feature)),dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Willingness to Accept Decision") + 
  ylab("") +
  ggtitle("") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

#### Figure K5 ####

mm_by_political_extremes_accept <- cj(accept ~ 
                                        consult + 
                                        council+
                                        ep + 
                                        eu_opinion +
                                        national_opinion +
                                        congruence, data = eudata_long, id = ~UUID, 
                                      estimate = "mm", by = ~political_extremes)

names(mm_by_political_extremes_accept)[names(mm_by_political_extremes_accept)=="level"]="term"
names(mm_by_political_extremes_accept)[names(mm_by_political_extremes_accept)=="political_extremes"]="model"

mm_by_political_extremes_accept$feature <- mm_by_political_extremes_accept$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_political_extremes_accept$feature <- factor(mm_by_political_extremes_accept$feature, levels = c("Egotropic representation",
                                                                                                      "Sociotropic representation (EU-wide)",
                                                                                                      "Sociotropic representation (national)",
                                                                                                      "Consultation",
                                                                                                      "Voting in Council",
                                                                                                      "Voting in EP"))
mm_by_political_extremes_accept <- mm_by_political_extremes_accept %>% filter(grepl("Sociotropic",feature))

# Figure K5
dwplot(mm_by_political_extremes_accept,dot_args = list(size = 2,aes(shape = model)),whisker_args = list(aes(linetype = model))) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Willingness to Accept Decision") + 
  ylab("") +
  ggtitle("") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme_results

#### Appendix L ####

#### Figure L1 ####

eudata_long$legit_diff_accept <- eudata_long$accept-eudata_long$legitimate

results_eu_legit_diff_accept <- cregg::mm(legit_diff_accept ~ 
                                            consult + 
                                            council+
                                            ep + 
                                            eu_opinion +
                                            national_opinion +
                                            congruence, data = eudata_long, id = ~UUID)

names(results_eu_legit_diff_accept)[4]="term"

results_eu_legit_diff_accept$feature <- results_eu_legit_diff_accept$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

results_eu_legit_diff_accept$feature <- factor(results_eu_legit_diff_accept$feature, levels = c("Egotropic representation",
                                                                                                "Sociotropic representation (EU-wide)",
                                                                                                "Sociotropic representation (national)",
                                                                                                "Consultation",
                                                                                                "Voting in Council",
                                                                                                "Voting in EP"))

# This is Figure L1
dwplot(results_eu_legit_diff_accept,dot_args = list(size = 2)) %>% 
  relabel_predictors(label_changes) +
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("More Legitimate ... More Willing to Accept") + 
  ylab("") +
  scale_color_manual(values = c("black")) +
  ggtitle("") + 
  theme_results

#### Appendix M ####

#### Figure M1 ####

store <- data.frame(decision_popularity = rep(seq(0,1,0.01),20),
                    legitimacy_change_socio = NA, legitimacy_change_ind = NA,
                    twentyfive_percent_socio = NA, twentyfive_percent_ind = NA,
                    stringsAsFactors = FALSE)

for(j in 1:nrow(store)){
  dat <- data.frame(person_id = 1:693,
                    country = c(rep("A",99),rep("B",99),rep("C",99),rep("D",99),
                                rep("E",99),rep("F",99),rep("G",99)),stringsAsFactors = F)
  
  for(k in 1:nrow(dat)){
    propensity.store = case_when(
      dat$country[k] =="A" ~ rbeta(1,10,1),
      dat$country[k] =="B" ~ rbeta(1,5,1),
      dat$country[k] =="C" ~ rbeta(1,2,1),
      dat$country[k] =="D" ~ rbeta(1,1,1),
      dat$country[k] =="E" ~ rbeta(1,1,2),
      dat$country[k] =="F" ~ rbeta(1,1,5),
      dat$country[k] =="G" ~ rbeta(1,1,10))
    dat$propensity[k] <- propensity.store
  }
  
  dat$pref <- ifelse(dat$propensity<store$decision_popularity[j],1,0)
  
  dat$legit_socio <- 0
  
  for(i in 1:nrow(dat)){
    country <- dat$country[i]
    country_maj <- ifelse(mean(dat$pref[dat$country==country])>0.5,1,0)
    overall_maj <- ifelse(mean(dat$pref)>0.5,1,0)
    legit=case_when(
      dat$pref[i] ==0 & country_maj ==0 & overall_maj ==0 ~  -0.77-0.085-0.095,
      dat$pref[i] ==0 & country_maj ==0 & overall_maj ==1 ~  -0.77-0.085+0.095,
      dat$pref[i] ==0 & country_maj ==1 & overall_maj ==0 ~  -0.77+0.085-0.095,
      dat$pref[i] ==0 & country_maj ==1 & overall_maj ==1 ~  -0.77+0.085+0.095,
      dat$pref[i] ==1 & country_maj ==0 & overall_maj ==0 ~   0.77-0.085-0.095,
      dat$pref[i] ==1 & country_maj ==0 & overall_maj ==1 ~   0.77-0.085+0.095,
      dat$pref[i] ==1 & country_maj ==1 & overall_maj ==0 ~   0.77+0.085-0.095,
      dat$pref[i] ==1 & country_maj ==1 & overall_maj ==1 ~   0.77+0.085+0.095
    )
    dat$legit_socio[i] <- legit
  }
  dat$legit_ind <- 0
  for(i in 1:nrow(dat)){
    country <- dat$country[i]
    country_maj <- ifelse(mean(dat$pref[dat$country==country])>0.5,1,0)
    overall_maj <- ifelse(mean(dat$pref)>0.5,1,0)
    legit=case_when(
      dat$pref[i] ==0 & country_maj ==0 & overall_maj ==0 ~  -0.77,
      dat$pref[i] ==0 & country_maj ==0 & overall_maj ==1 ~  -0.77,
      dat$pref[i] ==0 & country_maj ==1 & overall_maj ==0 ~  -0.77,
      dat$pref[i] ==0 & country_maj ==1 & overall_maj ==1 ~  -0.77,
      dat$pref[i] ==1 & country_maj ==0 & overall_maj ==0 ~   0.77,
      dat$pref[i] ==1 & country_maj ==0 & overall_maj ==1 ~   0.77,
      dat$pref[i] ==1 & country_maj ==1 & overall_maj ==0 ~   0.77,
      dat$pref[i] ==1 & country_maj ==1 & overall_maj ==1 ~   0.77
    )
    dat$legit_ind[i] <- legit
  }
  
  
  store$legitimacy_change_socio[j] <- mean(dat$legit_socio)
  store$twentyfive_percent_socio[j] <- mean(sort(dat$legit_socio)[1:round(0.25*nrow(dat),0)])
  
  store$legitimacy_change_ind[j] <- mean(dat$legit_ind)
  store$twentyfive_percent_ind[j] <- mean(sort(dat$legit_ind)[1:round(0.25*nrow(dat),0)])
  
  print(j)
}


df <- reshape::melt(store, id.vars = "decision_popularity", measure.vars = c("twentyfive_percent_ind", "twentyfive_percent_socio"))
df$variable <- df$variable %>% 
  recode("twentyfive_percent_socio" = "Egotropic and \nSociotropic Effects",
         "twentyfive_percent_ind" = "Only Egotropic \nEffects")

df <- aggregate(value~decision_popularity+variable,df,mean)

plot_sim_25 <- ggplot(data=df,aes(x=decision_popularity,y=value,group=variable,
                                  color=variable,shape=variable))+
  scale_colour_grey(start = 0.25, end = .6) +
  geom_point()+
  labs(x="Share of Citizens that \nAgree With Decision",y="Change in Legitimacy")+
  theme_bw()+
  theme(plot.title = element_text(face="bold"),
        axis.text=element_text(colour="black"),
        axis.title=element_text(colour="black"),
        strip.text.y = element_text(angle = 0, face="bold"),
        strip.background = element_blank(),
        panel.border = element_blank(), 
        axis.ticks = element_blank(),
        legend.position = "bottom",
        legend.title = element_blank())

df_comb <- df %>% 
  filter(variable=="Only Egotropic \nEffects") %>% 
  rename("value_ego" = "value") %>% 
  select(-variable) %>% 
  left_join(df %>% 
              filter(variable=="Egotropic and \nSociotropic Effects") %>% 
              rename("value_socio" = "value") %>% 
              select(-variable))

df_comb$diff <- df_comb$value_socio- df_comb$value_ego

plot_sim_25_diff <- ggplot(data=df_comb,aes(x=decision_popularity,y=diff))+
  geom_point()+
  labs(x="Share of Citizens that \nAgree With Decision",y="Difference in Legitimacy Change")+
  scale_colour_grey(start = 0.25, end = .6) +
  theme_bw()+
  geom_hline(yintercept=0)+
  theme(plot.title = element_text(face="bold"),
        axis.text=element_text(colour="black"),
        axis.title=element_text(colour="black"),
        strip.text.y = element_text(angle = 0, face="bold"),
        strip.background = element_blank(),
        panel.border = element_blank(), 
        axis.ticks = element_blank(),
        legend.position = "bottom",
        legend.title = element_blank())

# This is Figure M1
gridExtra::grid.arrange(plot_sim_25,plot_sim_25_diff,ncol=2)

