####################
# Replication code for Wratil, Wäckerle: "Majority Representation and Legitimacy: Survey-Experimental Evidence from the European Union" ####
# General Information: All plots in the paper and the Appendix are done using 
# the tikzDevice package. This means that plots from this script displayed 
# in the preview window or saved to pdf/png might look slightly different, but
# only in terms of size of axis labels, headings and the like. The quantities 
# displayed are equivalent to the ones in the paper/appendix.

#### Intro ####

library(cregg)
library(tidyverse)
library(dotwhisker)
library(tikzDevice)
options(tikzDocumentDeclaration = "\\documentclass[12pt]{article}")

sessionInfo()

load("datasets_eu.RData")

# There are two datasets in the RData file: "eudata", which is the original 
# dataset with one observation per respondent and "eudata_long", which has
# one observation per answered vignette (each respondent answered 5 vignettes).

#### Figure 2a) ####

results_eu_main <- cregg::mm(legitimate ~ 
                               congruence+
                               consult + 
                               council+
                               ep + 
                               eu_opinion +
                               national_opinion, data = eudata_long, id = ~UUID)

names(results_eu_main)[4]="term"

results_eu_main$feature <- results_eu_main$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

results_eu_main$feature <- factor(results_eu_main$feature, levels = c("Egotropic representation",
                                                                      "Sociotropic representation (EU-wide)",
                                                                      "Sociotropic representation (national)",
                                                                      "Consultation",
                                                                      "Voting in Council",
                                                                      "Voting in EP"))

results_eu_main$lower_84 <- results_eu_main$estimate-1.44*results_eu_main$std.error
results_eu_main$upper_84 <- results_eu_main$estimate+1.44*results_eu_main$std.error

results_eu_main$term <- results_eu_main$term %>% 
  recode("<blank>" = "No",
         "In this instance, the European Commission asked EU citizens and interest groups to submit their views on the topic." = "Yes",
         "All national governments voted in favor." = "Unanimity",
         "Most national governments voted in favor including the [COUNTRY] government, some voted against the decision." = "Majority incl. government",
         "Most national governments voted in favor, some voted against the decision including the [COUNTRY] government." = "Majority not incl. government",
         "Most" = "Large majority",
         "Slightly more than half" = "Slight majority", 
         "Most EU citizens oppose this decision" = "No",
         "Most EU citizens support this decision" = "Yes", 
         " most [NATIONALITY] support it" = "Yes", 
         " most [NATIONALITY] oppose it" = "No")

# This is Figure 2a)
results_eu_main %>% 
  ggplot(aes(xmin=lower,xmax=upper,y=term,x=estimate))+
  geom_point(size=1.75)+
  geom_errorbarh(height=0,size=0.75)+
  geom_errorbarh(height=0,size=1.5,data=results_eu_main,aes(xmin=lower_84,xmax=upper_84,y=term))+
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("") + 
  scale_color_manual(values = c("black")) +
  theme(plot.title = element_text(face="bold"),
        legend.position = "none",
        axis.text=element_text(colour="black"),
        axis.title=element_text(size=8,colour="black"),
        strip.text.y = element_text(angle = 0, face="bold"),
        #          strip.background =element_rect(fill="white"),
        strip.background = element_blank(),
        panel.border = element_blank(), 
        axis.ticks = element_blank(),
        panel.background = element_blank(), 
        #panel.grid = element_blank(),          
        panel.spacing.x = unit(0,"line"))

#### Figure 2b) ####

mm_non_congruent <- cj(legitimate ~ 
                         consult + 
                         council+
                         ep + 
                         eu_opinion +
                         national_opinion , data = eudata_long, by = ~congruence, estimate = "mm", id = ~UUID)

names(mm_non_congruent)[names(mm_non_congruent)=="level"]="term"
names(mm_non_congruent)[names(mm_non_congruent)=="congruence"]="model"

mm_non_congruent$feature <- mm_non_congruent$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_non_congruent$feature <- factor(mm_non_congruent$feature, levels = c("Egotropic representation",
                                                                        "Sociotropic representation (EU-wide)",
                                                                        "Sociotropic representation (national)",
                                                                        "Consultation",
                                                                        "Voting in Council",
                                                                        "Voting in EP"))

my.labels <- c("Marginal Mean of \nThinking Decision Is Legitimate")

mm_non_congruent$lower_84 <- mm_non_congruent$estimate-1.44*mm_non_congruent$std.error
mm_non_congruent$upper_84 <- mm_non_congruent$estimate+1.44*mm_non_congruent$std.error

mm_non_congruent$term <- mm_non_congruent$term %>% 
  recode("<blank>" = "No",
         "In this instance, the European Commission asked EU citizens and interest groups to submit their views on the topic." = "Yes",
         "All national governments voted in favor." = "Unanimity",
         "Most national governments voted in favor including the [COUNTRY] government, some voted against the decision." = "Majority incl. government",
         "Most national governments voted in favor, some voted against the decision including the [COUNTRY] government." = "Majority not incl. government",
         "Most" = "Large majority",
         "Slightly more than half" = "Slight majority", 
         "Most EU citizens oppose this decision" = "No",
         "Most EU citizens support this decision" = "Yes", 
         " most [NATIONALITY] support it" = "Yes", 
         " most [NATIONALITY] oppose it" = "No")


plot_congruent <- mm_non_congruent  %>% filter(model=="Yes") %>%
  ggplot(aes(xmin=lower,xmax=upper,y=term,x=estimate))+
  geom_point(size=1.75)+
  geom_errorbarh(height=0,size=0.75)+
  geom_errorbarh(height=0,size=1.5,data=mm_non_congruent%>% filter(model=="Yes"),aes(xmin=lower_84,xmax=upper_84,y=term))+
  ggtitle("Egotropically \nCongruent Decisions")+
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking \nDecision Is Legitimate") + 
  ylab("") +
  ggtitle("Egotropically \nCongruent Decisions")+
  scale_color_manual(values = c("black")) +
  theme(plot.title = element_text(face="bold",size=10),
        axis.text=element_text(colour="black"),
        axis.title=element_text(size=8,colour="black"),
        strip.text.y = element_text(angle = 0, face="bold"),
        #          strip.background =element_rect(fill="white"),
        strip.background = element_blank(),
        panel.border = element_blank(), 
        axis.ticks = element_blank(),
        panel.background = element_blank(), 
        #panel.grid = element_blank(),          
        panel.spacing.x = unit(0,"line"),
        legend.position = "none",
        legend.title = element_blank())
plot_noncongruent <- mm_non_congruent  %>% filter(model=="No") %>%
  ggplot(aes(xmin=lower,xmax=upper,y=term,x=estimate))+
  geom_point(size=1.75)+
  geom_errorbarh(height=0,size=0.75)+
  geom_errorbarh(height=0,size=1.5,data=mm_non_congruent%>% filter(model=="No"),aes(xmin=lower_84,xmax=upper_84,y=term))+
  ggtitle("Egotropically \nIncongruent Decisions")+
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking \nDecision Is Legitimate") + 
  ylab("") +
  scale_color_manual(values = c("black")) +
  theme(plot.title = element_text(face="bold",size=10),
        axis.text=element_text(colour="black"),
        axis.title=element_text(size=8,colour="black"),
        strip.text.y = element_blank(),
        #          strip.background =element_rect(fill="white"),
        strip.background = element_blank(),
        panel.border = element_blank(), 
        axis.ticks = element_blank(),
        panel.background = element_blank(), 
        #panel.grid = element_blank(),          
        panel.spacing.x = unit(0,"line"),
        legend.position = "none",
        legend.title = element_blank())

# This is Figure 2b)
a <- egg::ggarrange(plot_congruent + 
                      theme(axis.ticks.y = element_blank(),
                            plot.margin = margin(r = 1, l = 1) ),
                    plot_noncongruent +
                      theme(axis.text.y = element_blank(),
                            axis.ticks.y = element_blank(),
                            axis.title.y = element_blank(),
                            plot.margin = margin(r = 1) ), 
                    nrow = 1)

#### Figure 3a) ####

eudata_long$EU_identity_rec=recode(eudata_long$EU_identity,
                                   "[COUNTRY adj] and European" = "Some European Identity",
                                   "[COUNTRY adj] only" = "Exclusive National Identity",
                                   "European and [COUNTRY adj]" = "Some European Identity",
                                   "European only" = "Some European Identity")

eudata_identity <- eudata_long %>% filter(EU_identity_rec != "None of the above")
eudata_identity$EU_identity_rec <- as.factor(eudata_identity$EU_identity_rec)
mm_by_identity <- cj(legitimate ~ 
                       consult + 
                       council+
                       ep + 
                       eu_opinion +
                       national_opinion +
                       congruence, data = eudata_identity, id = ~UUID, 
                     estimate = "mm", by = ~EU_identity_rec)

names(mm_by_identity)[names(mm_by_identity)=="level"]="term"
names(mm_by_identity)[names(mm_by_identity)=="EU_identity_rec"]="model"

mm_by_identity$feature <- mm_by_identity$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_identity$feature <- factor(mm_by_identity$feature, levels = c("Egotropic representation",
                                                                    "Sociotropic representation (EU-wide)",
                                                                    "Sociotropic representation (national)",
                                                                    "Consultation",
                                                                    "Voting in Council",
                                                                    "Voting in EP"))

mm_by_identity$lower_84 <- mm_by_identity$estimate-1.44*mm_by_identity$std.error
mm_by_identity$upper_84 <- mm_by_identity$estimate+1.44*mm_by_identity$std.error

mm_by_identity$term <- mm_by_identity$term %>% 
  recode("<blank>" = "No",
         "In this instance, the European Commission asked EU citizens and interest groups to submit their views on the topic." = "Yes",
         "All national governments voted in favor." = "Unanimity",
         "Most national governments voted in favor including the [COUNTRY] government, some voted against the decision." = "Majority incl. government",
         "Most national governments voted in favor, some voted against the decision including the [COUNTRY] government." = "Majority not incl. government",
         "Most" = "Large majority",
         "Slightly more than half" = "Slight majority", 
         "Most EU citizens oppose this decision" = "No",
         "Most EU citizens support this decision" = "Yes", 
         " most [NATIONALITY] support it" = "Yes", 
         " most [NATIONALITY] oppose it" = "No")

mm_by_identity_opinion_only = mm_by_identity %>% filter(grepl("Sociotropic",feature))

mm_by_identity_opinion_only$model <- mm_by_identity_opinion_only$model %>% 
  recode("Exclusive National Identity" = "Exclusive National \nIdentity",
         "Some European Identity" = "Some European \nIdentity")

# This is Figure 3a)
mm_by_identity_opinion_only %>%
  ggplot(aes(xmin=lower,xmax=upper,y=term,x=estimate,color=model,shape=model))+
  geom_point(size=1.75,position=position_dodge(width = 0.6))+
  geom_errorbarh(height=0,size=0.75,position=position_dodge(width = 0.6))+
  geom_errorbarh(height=0,
                 size=1.5,
                 data=mm_by_identity_opinion_only,
                 aes(xmin=lower_84,xmax=upper_84,y=term),
                 position=position_dodge(width = 0.6))+
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("a) Group Endorsement") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme(plot.title = element_text(face="bold"),
        axis.text=element_text(colour="black"),
        axis.title=element_text(size=8,colour="black"),
        strip.text.y = element_text(angle = 0, face="bold"),
        #          strip.background =element_rect(fill="white"),
        strip.background = element_blank(),
        panel.border = element_blank(), 
        axis.ticks = element_blank(),
        panel.background = element_blank(), 
        #panel.grid = element_blank(),          
        panel.spacing.x = unit(0,"line"),
        legend.position = "bottom",
        legend.title = element_blank())

#### Figure 3b) ####

eudata_long = eudata_long %>%
  mutate(
    topic_impression_rec = case_when(
      topic_impression <= 38                           ~ "First quartile",
      topic_impression > 38 & topic_impression <= 52   ~ "Second quartile",
      topic_impression > 52 & topic_impression <= 79   ~ "Third quartile",
      topic_impression > 79                            ~ "Fourth quartile"
    )
  )
eudata_long = eudata_long %>%
  mutate(
    topic_impression_rec_median = case_when(
      topic_impression <= 52                           ~ "Strong Issue Attitudes",
      topic_impression > 52   ~ "Weak Issue Attitudes"
    )
  )

eudata_long$topic_impression_rec_median <- as.factor(eudata_long$topic_impression_rec_median)
mm_by_topicimpression=cj(eudata_long, legitimate ~ 
                           consult + 
                           council+
                           ep + 
                           eu_opinion +
                           national_opinion +
                           congruence, id = ~UUID, by = ~topic_impression_rec_median, estimate = "mm")

names(mm_by_topicimpression)[names(mm_by_topicimpression)=="level"]="term"
names(mm_by_topicimpression)[names(mm_by_topicimpression)=="topic_impression_rec_median"]="model"

mm_by_topicimpression$feature <- mm_by_topicimpression$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_topicimpression$feature <- factor(mm_by_topicimpression$feature, levels = c("Egotropic representation",
                                                                                  "Sociotropic representation (EU-wide)",
                                                                                  "Sociotropic representation (national)",
                                                                                  "Consultation",
                                                                                  "Voting in Council",
                                                                                  "Voting in EP"))

mm_by_topicimpression$lower_84 <- mm_by_topicimpression$estimate-1.44*mm_by_topicimpression$std.error
mm_by_topicimpression$upper_84 <- mm_by_topicimpression$estimate+1.44*mm_by_topicimpression$std.error

mm_by_topicimpression$term <- mm_by_topicimpression$term %>% 
  recode("<blank>" = "No",
         "In this instance, the European Commission asked EU citizens and interest groups to submit their views on the topic." = "Yes",
         "All national governments voted in favor." = "Unanimity",
         "Most national governments voted in favor including the [COUNTRY] government, some voted against the decision." = "Majority incl. government",
         "Most national governments voted in favor, some voted against the decision including the [COUNTRY] government." = "Majority not incl. government",
         "Most" = "Large majority",
         "Slightly more than half" = "Slight majority", 
         "Most EU citizens oppose this decision" = "No",
         "Most EU citizens support this decision" = "Yes", 
         " most [NATIONALITY] support it" = "Yes", 
         " most [NATIONALITY] oppose it" = "No")

mm_by_topicimpression_opinion_only = mm_by_topicimpression %>% filter(grepl("Sociotropic",feature))

mm_by_topicimpression_opinion_only$model <- mm_by_topicimpression_opinion_only$model %>% 
  recode("Strong Issue Attitudes" = "Strong Issue \nAttitudes",
         "Weak Issue Attitudes" = "Weak Issue \nAttitudes")

# This is Figure 3b)
mm_by_topicimpression_opinion_only %>%
  ggplot(aes(xmin=lower,xmax=upper,y=term,x=estimate,color=model,shape=model))+
  geom_point(size=1.75,position=position_dodge(width = 0.6))+
  geom_errorbarh(height=0,size=0.75,position=position_dodge(width = 0.6))+
  geom_errorbarh(height=0,
                 size=1.5,
                 data=mm_by_topicimpression_opinion_only,
                 aes(xmin=lower_84,xmax=upper_84,y=term),
                 position=position_dodge(width = 0.6))+
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("b) Consensus Heuristic") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme(plot.title = element_text(face="bold"),
        axis.text=element_text(colour="black"),
        axis.title=element_text(size=8,colour="black"),
        strip.text.y = element_text(angle = 0, face="bold"),
        #          strip.background =element_rect(fill="white"),
        strip.background = element_blank(),
        panel.border = element_blank(), 
        axis.ticks = element_blank(),
        panel.background = element_blank(), 
        #panel.grid = element_blank(),          
        panel.spacing.x = unit(0,"line"),
        legend.position = "bottom",
        legend.title = element_blank())


#### Figure 3c) ####
eudata_long <- eudata_long %>% 
  mutate(political_extremes = case_when(
    lire %in% c(0,1,2,8,9,10) ~ "Ideological Extremists",
    TRUE  ~ "Moderates"
  ))

eudata_long$political_extremes <- as.factor(eudata_long$political_extremes)
mm_by_political_extremes <- cj(legitimate ~ 
                                 consult + 
                                 council+
                                 ep + 
                                 eu_opinion +
                                 national_opinion +
                                 congruence, data = eudata_long, id = ~UUID, 
                               estimate = "mm", by = ~political_extremes)

names(mm_by_political_extremes)[names(mm_by_political_extremes)=="level"]="term"
names(mm_by_political_extremes)[names(mm_by_political_extremes)=="political_extremes"]="model"

mm_by_political_extremes$feature <- mm_by_political_extremes$feature %>% 
  recode("congruence" = "Egotropic representation",
         "consult" = "Consultation",
         "council" = "Voting in Council",
         "ep" = "Voting in EP",
         "eu_opinion" = "Sociotropic representation (EU-wide)",
         "national_opinion" = "Sociotropic representation (national)")

mm_by_political_extremes$feature <- factor(mm_by_political_extremes$feature, levels = c("Egotropic representation",
                                                                                        "Sociotropic representation (EU-wide)",
                                                                                        "Sociotropic representation (national)",
                                                                                        "Consultation",
                                                                                        "Voting in Council",
                                                                                        "Voting in EP"))
mm_by_political_extremes <- mm_by_political_extremes %>% filter(grepl("Sociotropic",feature))

mm_by_political_extremes$model <- mm_by_political_extremes$model %>% 
  recode("Ideological Extremists" = "Ideological \nExtremists")

mm_by_political_extremes$lower_84 <- mm_by_political_extremes$estimate-1.44*mm_by_political_extremes$std.error
mm_by_political_extremes$upper_84 <- mm_by_political_extremes$estimate+1.44*mm_by_political_extremes$std.error

mm_by_political_extremes$term <- mm_by_political_extremes$term %>% 
  recode("<blank>" = "No",
         "In this instance, the European Commission asked EU citizens and interest groups to submit their views on the topic." = "Yes",
         "All national governments voted in favor." = "Unanimity",
         "Most national governments voted in favor including the [COUNTRY] government, some voted against the decision." = "Majority incl. government",
         "Most national governments voted in favor, some voted against the decision including the [COUNTRY] government." = "Majority not incl. government",
         "Most" = "Large majority",
         "Slightly more than half" = "Slight majority", 
         "Most EU citizens oppose this decision" = "No",
         "Most EU citizens support this decision" = "Yes", 
         " most [NATIONALITY] support it" = "Yes", 
         " most [NATIONALITY] oppose it" = "No")

# This is Figure 3c)
mm_by_political_extremes %>%
  ggplot(aes(xmin=lower,xmax=upper,y=term,x=estimate,color=model,shape=model))+
  geom_point(size=1.75,position=position_dodge(width = 0.6))+
  geom_errorbarh(height=0,size=0.75,position=position_dodge(width = 0.6))+
  geom_errorbarh(height=0,
                 size=1.5,
                 data=mm_by_political_extremes,
                 aes(xmin=lower_84,xmax=upper_84,y=term),
                 position=position_dodge(width = 0.6))+
  facet_grid(feature~.,scales = "free",space = "free",labeller = label_wrap_gen(width=20))+
  theme_bw() + 
  xlab("Marginal Mean of Thinking Decision Is Legitimate") + 
  ylab("") +
  ggtitle("c) Long-term Utilitarian Calculus") + 
  scale_colour_grey(start = 0.25, end = .6) +
  theme(plot.title = element_text(face="bold"),
        axis.text=element_text(colour="black"),
        axis.title=element_text(size=8,colour="black"),
        strip.text.y = element_text(angle = 0, face="bold"),
        #          strip.background =element_rect(fill="white"),
        strip.background = element_blank(),
        panel.border = element_blank(), 
        axis.ticks = element_blank(),
        panel.background = element_blank(), 
        #panel.grid = element_blank(),          
        panel.spacing.x = unit(0,"line"),
        legend.position = "bottom",
        legend.title = element_blank())
