/* 

Replication syntax EJPR article
"Political socialization, political gender gaps, and the intergenerational transmission of left-right ideology"
Dr. Mathilde M. van Ditmars, University of Lucerne, February 2022 (Stata version 17)

*/

************
***G-SOEP***
************

*German Socio-Economic Panel (SOEP), data of the years 1984-2013
*Data version used: G-SOEP data v30; DOI: 10.5684/soep.v30
*see https://www.diw.de/sixcms/detail.php?id=742059


*DATASET WORKING FILE CREATION*

global workdir "H:\Datafilecreation\v30_new"

cd "Z:\DATA\soep30_en_l\stata"

*take variables from the long files
*pl
use cid syear hid pid plh0004 plh0007 plh0011 plh0012 plh0013 plh0033 plh0016 ///
plh0019 plh0021 plh0023 plh0025 using pl, clear
rename plh0004 lire
rename plh0007 polint
rename plh0011 partyid
rename plh0012 pidpref
rename plh0013 pidstr
rename plh0033 fin_worries
rename plh0016 finsec_fam
rename plh0019 finsec_unemp
rename plh0021 finsec_sick
rename plh0023 finsec_old
rename plh0025 finsec_care
sort pid
save $workdir\pl_selection, replace



*merge with pgen variables
use $workdir\pl_selection, clear
sort pid
merge 1:1 pid syear using pgen, keepusing (pgisei pgsiops pgisced ///
pgisco08 pgfamstd pgegp) keep (match) nogen
sort pid
save $workdir\longfile_children, replace

*merge with ppfad variables 
use $workdir\longfile_children, clear
sort pid
merge 1:1 pid syear using ppfadl, keepusing (loc1989 sex gebjahr) keep (match) nogen
sort pid
save $workdir\longfile_children_ppfadl, replace

*make long file for parents
*1. mother
use $workdir\longfile_children, clear
renvars _all, postfix(_m)
rename pid_m mnr
sort mnr
save $workdir\longfile_mother, replace

*2. father
use $workdir\longfile_children, clear
renvars _all, postfix(_v)
rename pid_v vnr
sort vnr
save $workdir\longfile_father, replace

cd "Z:\DATA\soep30_en\stata"

*using cross-sectional data to create working file

*bioparen
use persnr hhnr mreli vreli vsiops msiops vnr mnr mtodj vtodj living1-living8 vbstell mbstell vbsinfo mbsinfo geschw ///
nums numb using bioparen, clear
sort persnr
save $workdir\bioparen_selection

*biosib
use $workdir\bioparen_selection, clear
merge 1:1 persnr using biosib, keepusing (num_sib pos_sib sexsib1-sexsib11 ///
gebsib1-gebsib11 sibdef1-sibdef11) keep (master match) nogen
sort persnr
save $workdir\bioparen_biosib_selection, replace


cd "Z:\DATA\soep30_en_l\stata"
*add cross-sectional ppfad data
use $workdir\bioparen_biosib_selection, clear
rename persnr pid
sort pid
merge 1:1 pid using ppfad, keepusing (loc1989 sex gebjahr) keep(match) nogen
sort pid
save $workdir\crosssectfile_children, replace

*reshape long children file to wide and match to cross-sect. file
use $workdir\longfile_children, clear

*before reshaping, preserve the variable labels
foreach v of var*{
        local l`v' : variable label `v'
 }
local list "syear lire polint partyid pidpref pidstr fin_worries finsec_fam finsec_unemp finsec_sick finsec_old finsec_care pgisei pgsiops pgisced pgisco08 pgfamstd pgegp"

 /* save the value labels for variables in local list*/
 foreach var of local list{
	 levelsof `var', local(`var'_levels)       /* create local list of all values of `var' */
	 foreach val of local `var'_levels {       /* loop over all values in local list `var'_levels */
      	 local `var'vl`val' : label `var' `val'    /* create macro that contains label for each value */
       }
 }


*reshape
reshape wide cid hid lire polint partyid pidpref pidstr fin_worries finsec_fam /// 
finsec_unemp finsec_sick finsec_old finsec_care pgisei pgsiops pgisced ///
pgisco08 pgfamstd, i(pid) j(syear)
sort pid

local variablelist "lire polint partyid pidpref pidstr fin_worries finsec_fam finsec_unemp finsec_sick finsec_old finsec_care pgisei pgsiops pgisced pgisco08 pgfamstd pgegp"

 /* apply the variable & value labels as variable labels */
 /* variables are in form answeryear incyear */
 foreach variable of local variablelist{                 /* loop over list "inc answer" */
	 foreach value of local year_levels{             /* loop over list "80 81 82" */
		 label variable `variable'`value' "`l`variable'': `yearvl`value''"
	 }
}

merge 1:1 pid using $workdir\crosssectfile_children, keep (match) nogen
sort pid
rename pid persnr
save $workdir\workingfile_children, replace

*reshape long parents' files to wide

***MOTHER 
use $workdir\longfile_mother, clear

foreach v of var*{
        local l`v' : variable label `v'
 }
local list "syear_m lire_m polint_m partyid_m pidpref_m pidstr_m fin_worries_m finsec_fam_m finsec_unemp_m finsec_sick_m finsec_old_m finsec_care_m pgisei_m pgsiops_m pgisced_m pgisco08_m pgfamstd_m pgegp_m"

 /* save the value labels for variables in local list*/
 foreach var of local list{
	 levelsof `var', local(`var'_levels)       /* create local list of all values of `var' */
	 foreach val of local `var'_levels {       /* loop over all values in local list `var'_levels */
      	 local `var'vl`val' : label `var' `val'    /* create macro that contains label for each value */
       }
 }



reshape wide cid_m hid_m lire_m polint_m partyid_m pidpref_m pidstr_m fin_worries_m finsec_fam_m /// 
finsec_unemp_m finsec_sick_m finsec_old_m finsec_care_m pgisei_m pgsiops_m pgisced_m ///
pgisco08_m pgfamstd_m, i(mnr) j(syear)


local variablelist "lire_m polint_m partyid_m pidpref_m pidstr_m fin_worries_m finsec_fam_m finsec_unemp_m finsec_sick_m finsec_old_m finsec_care_m pgisei_m pgsiops_m pgisced_m pgisco08_m pgfamstd_m pgegp_m"

 /* apply the variable & value labels as variable labels */
 /* variables are in form answeryear incyear */
 foreach variable of local variablelist{                 /* loop over list "inc answer" */
	 foreach value of local year_levels{             /* loop over list "80 81 82" */
		 label variable `variable'`value' "`l`variable'': `yearvl`value''"
	 }
}
sort mnr
save $workdir\widefile_mother, replace

***FATHER
use $workdir\longfile_father, clear

foreach v of var*{
        local l`v' : variable label `v'
 }
local list "syear_v lire_v polint_v partyid_v pidpref_v pidstr_v fin_worries_v finsec_fam_v finsec_unemp_v finsec_sick_v finsec_old_v finsec_care_v pgisei_v pgsiops_v pgisced_v pgisco08_v pgfamstd_v pgegp_v"

 /* save the value labels for variables in local list*/
 foreach var of local list{
	 levelsof `var', local(`var'_levels)       /* create local list of all values of `var' */
	 foreach val of local `var'_levels {       /* loop over all values in local list `var'_levels */
      	 local `var'vl`val' : label `var' `val'    /* create macro that contains label for each value */
       }
 }

reshape wide cid_v hid_v lire_v polint_v partyid_v pidpref_v pidstr_v fin_worries_v finsec_fam_v /// 
finsec_unemp_v finsec_sick_v finsec_old_v finsec_care_v pgisei_v pgsiops_v pgisced_v ///
pgisco08_v pgfamstd_v, i(vnr) j(syear)

local variablelist "lire_v polint_v partyid_v pidpref_v pidstr_v fin_worries_v finsec_fam_v finsec_unemp_v finsec_sick_v finsec_old_v finsec_care_v pgisei_v pgsiops_v pgisced_v pgisco08_v pgfamstd_v pgegp_v"

 /* apply the variable & value labels as variable labels */
 /* variables are in form answeryear incyear */
 foreach variable of local variablelist{                 /* loop over list "inc answer" */
	 foreach value of local year_levels{             /* loop over list "80 81 82" */
		 label variable `variable'`value' "`l`variable'': `yearvl`value''"
	 }
}

sort vnr
save $workdir\widefile_father, replace

*match parental files to children's file
use $workdir\workingfile_children, clear
merge m:1 mnr using $workdir\widefile_mother, keep(master match) nogen
save $workdir\workingfile_children_mother, replace


merge m:1 vnr using $workdir\widefile_father, keep (master match) nogen
save $workdir\workingfile_children_parents, replace

*merge with regional data
sort persnr

capture drop _merge

merge 1:1 persnr using $workdir\vpequiv_small.dta
drop if _merge==2

rename _merge merge_vpequiv

merge 1:1 persnr using $workdir\zpequiv_small.dta

drop if _merge==2

save $workdir\workingfile_children_parents.dta, replace




*RECODING*

use $workdir\workingfile_children_parents.dta

recode all (-1=.a)(-2=.b)(-3=.c)(-4=.d)(-5=.e)(-6=.f)(-7=.g)(-8=.h) 
/* 
-1 	no answer / donít know
-2 	does not apply
-3 	implausible value
-4 	Inadmissible multiple response
-5 	Not included in this version of the questionnaire
-6 	Version of questionnaire with modified filtering
-8 	Question not part of the survey program this year
*/ 


*order variables

order _all, alphabetic
order hhnr persnr mnr vnr sex gebjahr, first

***left-right of children and parents
// not observed in the same year, but parental obs always before the child's

drop lire1984-lire2004
drop lire2006-lire2008
drop lire2010-lire2013

drop lire_m1984-lire_m2004
drop lire_m2006-lire_m2008
drop lire_m2010-lire_m2013

drop lire_v1984-lire_v2004
drop lire_v2006-lire_v2008
drop lire_v2010-lire_v2013

tab1 lire2005 lire2009
tab1 lire_m2005 lire_m2009
tab1 lire_v2005 lire_v2009

*offspring, baseline 2009
gen lire0509=lire2009
replace lire0509=lire2005 if lire2009>=. & lire2005<.
tab lire0509


*parents
gen lire0509m=lire_m2009 if lire2009<.
replace lire0509m=lire_m2005 if lire0509m>=. & lire2005<. 
tab lire0509m


gen lire0509v=lire_v2009 if lire2009<.
replace lire0509v=lire_v2005 if lire0509v>=. & lire2005<. 
tab lire0509v

*left-right in 3 blocks
recode lire0509 lire0509m lire0509v (0/4=1) (5=2) (6/10=3), gen (lire0509_3 lire0509m_3 lire0509v_3)
label define lire3 1 "left" 2 "centre" 3 "right"
label values lire0509_3 lire0509m_3 lire0509v_3 lire3
tab1 lire0509_3 lire0509m_3 lire0509v_3

*corresponding parental homogeneity measure
gen par_hom=.
replace par_hom=1 if (lire0509m_3==lire0509v_3) & lire0509m_3<. & lire0509v_3<.
replace par_hom=0 if (lire0509m_3!=lire0509v_3) & lire0509m_3<. & lire0509v_3<.
tab par_hom

*left/right-wing ideology
recode lire0509_3 (1=1) (2=0) (3=0), gen(left)
label define left 1 "left" 0 "centre/right"
label values left left
tab1 lire0509_3 left

recode lire0509_3 (1=0) (2=0) (3=1), gen(right)
label define right 1 "right" 0 "left/centre"
label values right right
tab1 lire0509_3 right

*centre category (for t-testing)
recode lire0509_3 (1=0) (2=1) (3=0), gen(centre)
label define centre 1 "centre" 0 "left/right"
label values centre centre
tab1 lire0509_3 centre

*left and right categories also for parents
recode lire0509m_3 (1=1) (2=0) (3=0), gen(left_m)
label values left_m left
tab1 lire0509m_3 left_m

recode lire0509m_3 (1=0) (2=0) (3=1), gen(right_m)
label values right_m right
tab1 lire0509m_3 right_m

recode lire0509v_3 (1=1) (2=0) (3=0), gen(left_v)
label values left_v left
tab1 lire0509v_3 left_v

recode lire0509v_3 (1=0) (2=0) (3=1), gen(right_v)
label values right_v right
tab1 lire0509v_3 right_v

*create new value labels for parental ideology
gen lire0509m_3_i=lire0509m_3
tab1 lire0509m_3 lire0509m_3_i
label define lire3m 1 "Mother Left" 2 "Mother Centre" 3 "Mother Right"
label values lire0509m_3_i lire3m
tab lire0509m_3_i

gen lire0509v_3_i=lire0509v_3
tab1 lire0509v_3 lire0509v_3_i
label define lire3v 1 "Father Left" 2 "Father Centre" 3 "Father Right"
label values lire0509v_3_i lire3v
tab lire0509v_3_i

*create dummies for parental ideology
tab lire0509m_3, gen(lire_m_dum)
tab lire0509v_3, gen(lire_v_dum)

*combinations of parental ideology (for descriptives)
gen par_combi=.
replace par_combi=1 if lire0509m_3==1 & lire0509v_3==1 // both left
replace par_combi=2 if lire0509m_3==1 & lire0509v_3==2 // mother left, father centre
replace par_combi=3 if lire0509m_3==1 & lire0509v_3==3 // mother left, father right

replace par_combi=4 if lire0509m_3==3 & lire0509v_3==1 // father left, mother right
replace par_combi=5 if lire0509m_3==2 & lire0509v_3==1 // father left, mother centre

replace par_combi=6 if lire0509m_3==3 & lire0509v_3==3 // both right
replace par_combi=7 if lire0509m_3==3 & lire0509v_3==2 // father centre, mother right
replace par_combi=8 if lire0509m_3==2 & lire0509v_3==3 // mother centre, father right
replace par_combi=0 if lire0509m_3==2 & lire0509v_3==2 // both centre 

label define par_combi 0 "both centre" 1 "both left" 2 "M left, F centre" 3 "M left, F right" ///
4 "F left, M right" 5 "F left, M centre" 6 "both right" 7 "F centre, M right" 8 "M centre, F right" 0 "both centre"

label values par_combi par_combi
tab par_combi

*similarity in ideological block parent-child
gen mc_hom=.
replace mc_hom=1 if (lire0509m_3==lire0509_3) & lire0509m_3<. & lire0509_3<.
replace mc_hom=0 if (lire0509m_3!=lire0509_3) & lire0509m_3<. & lire0509_3<.
tab mc_hom

gen vc_hom=.
replace vc_hom=1 if (lire0509v_3==lire0509_3) & lire0509v_3<. & lire0509_3<.
replace vc_hom=0 if (lire0509v_3!=lire0509_3) & lire0509v_3<. & lire0509_3<.
tab vc_hom


***female
tab sex
recode sex (1=0) (2=1), gen(female)
tab female
label define female 0 "male" 1 "female"
label values female female
tab female

***age in year of L/R self-placement
tab gebjahr
gen age0509=.
replace age0509=(2009-gebjahr) if lire2009<.
replace age0509=(2005-gebjahr) if lire2009>=. & lire2005<.
tab age0509

***isced: in year of L/R self-placement
gen isced0509=.
replace isced0509=pgisced2009 if lire2009<.
replace isced0509=pgisced2005 if lire2009>=. & lire2005<.

recode isced0509 (0=1 "in school") (1/2=2 "low") (3/4=3 "medium") (5/6=4 "high"), gen(isced0509_3)
tab isced0509_3

***father's SIOPS when 15
// look at 30th percentile and 70th percentile for cut-off scores
// siops=low if parents were unemployed (also due to sickness)
tab1 vsiops msiops
recode vsiops (min/33=1 "low") (34/45=2 "medium") (46/max=3 "high"), gen (siops_father)
replace siops_father=1 if vbstell==10 | vbstell==12
tab siops_father

***region DE 1989
capture drop region05 region09
clonevar region05= l1110205
clonevar region09= l1110209

tab region05 if lire0509<.

gen region0509 = region09 if lire2009<.
replace region0509=region05 if lire2009>=. & lire2005<.

tab region0509

recode region0509 (1=2 "west germany") (2=1 "east germany"), gen(loc0509)
tab loc0509

recode loc1989 (1=1 "East Germany") (2=2 "West Germany") (3=3 "Abroad"), gen(region)
replace region=loc0509 if loc1989==.a | loc1989==.b
tab1 loc1989 region


*include year dummy
gen year_lr=.
replace year_lr=2009 if lire2009<.
replace year_lr=2005 if lire2009>=. & lire2005<.

save workingfile_children_parents_recoded.dta, replace





*ANALYSIS

use workingfile_children_parents_recoded.dta

*Descriptive analysis with analytic sample
 
quietly regress lire0509 ib2.lire0509m_3_i ib2.lire0509v_3_i female ib2.siops_father i.isced0509_3 ///
age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36

* Distance in left-right ideology 
tab lireabdifm_3 if e(sample) 
tab lireabdifv_3 if e(sample) 

tab lireabdifm_3, gen(liredifm_dum)
tab lireabdifv_3, gen(liredifv_dum)

ttest liredifm_dum1=liredifv_dum1 if e(sample)

summarize lireabdif_m, if e(sample)
summarize lireabdif_v, if e(sample)

*correlation
pwcorr lire0509 lire0509m if e(sample) 
pwcorr lire0509 lire0509v if e(sample) 

*Table 1 and 2: Overlap in ideological blocks
tab lire0509_3 lire0509m_3, col, if e(sample)
tab lire0509_3 lire0509v_3, col, if e(sample)

*t-test 
ttest right=centre if lire0509v_3==3 & e(sample)
ttest right=centre if lire0509m_3==3 & e(sample)

* Table 3: Parental similarity in ideology
tab lire0509m_3 lire0509v_3, col, if e(sample)

tab lire_m_5 lire_v_5, summarize(lire0509) means freq, if e(sample)

*gender gap in ideology in parental and offspring generation
ttest lire0509, by(female),  if e(sample)

ttest lire0509m == lire0509v, unpaired, if e(sample)

*descriptives mothers
summarize lire0509 if e(sample) 
summarize lire0509m if e(sample) 
tab1 left left_m if e(sample)
tab lire0509_3 if e(sample)
tab lire0509m_3 if e(sample) 
ttest lire0509=lire0509m if e(sample)

*descriptives fathers
summarize lire0509 if e(sample) 
summarize lire0509v if e(sample) 
tab1 left left_v if e(sample)
tab lire0509_3 if e(sample) 
tab lire0509v_3 if e(sample)
ttest lire0509=lire0509v if e(sample)


tab lire0509_3 par_combi, col, if par_hom==1 &  e(sample)

tab lire0509_3 par_combi, col, if par_hom==0 & e(sample)

*postestimation
quietly regress vc_hom ib3.lire0509v_3##i.female par_hom ib2.siops_father ib2.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
predict vchom_pred_de
tab vchom_pred_de

quietly regress mc_hom ib3.lire0509m_3##i.female par_hom ib2.siops_father ib2.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
predict mchom_pred_de
tab mchom_pred_de


*Regressions


*Figure 1 / Table 4: probability of being in the same block


//MOTHERS
*1. baseline model
regress mc_hom ib3.lire0509m_3 par_hom ib2.siops_father ib2.isced0509_3 female age0509 i.year_lr ib3.region,cluster(hhnr), if age0509<36
outreg2 using RR_GSOEP_LR_hom.xls, replace ctitle (Mothers 1)


*2. interaction with gender
regress mc_hom ib3.lire0509m_3##i.female par_hom ib2.siops_father ib2.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
outreg2 using RR_GSOEP_LR_hom.xls, append ctitle (Mothers 2)

margins ib3.lire0509m_3, over(female) post

**compare significance of difference between boys and girl for LW mother
quietly regress mc_hom ib3.lire0509m_3##i.female par_hom ib2.siops_father ib2.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
margins r.female, over(lire0509m_3) post

margins lire0509m_3, over(female) post
marginsplot, recast(scatter) 
graph save DE_mothers, replace

quietly regress mc_hom ib3.lire0509m_3##i.female par_hom ib2.siops_father ib2.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
margins lire0509m_3, over(female) post, if female==1
est store de_mhom_g

quietly regress mc_hom ib3.lire0509m_3##i.female par_hom ib2.siops_father ib2.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
margins lire0509m_3, over(female) post, if female==0
est store de_mhom_b




//FATHERS
*1. baseline model
regress vc_hom ib3.lire0509v_3 par_hom ib2.siops_father ib2.isced0509_3 female age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
outreg2 using RR_GSOEP_LR_hom.xls, append ctitle (Fathers 1)

*2. interaction with gender
regress vc_hom ib3.lire0509v_3##i.female par_hom ib2.siops_father ib2.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
outreg2 using RR_GSOEP_LR_hom.xls, append ctitle (Fathers 2)
margins r.female, over(lire0509v_3) post

quietly regress vc_hom ib3.lire0509v_3##i.female par_hom ib2.siops_father ib2.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
margins lire0509v_3, over(female) post
marginsplot, recast(scatter) horizontal

quietly regress vc_hom ib3.lire0509v_3##i.female par_hom ib2.siops_father ib2.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
margins lire0509v_3, over(female) post, if female==1
est store de_vhom_g

quietly regress vc_hom ib3.lire0509v_3##i.female par_hom ib2.siops_father ib2.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
margins lire0509v_3, over(female) post, if female==0
est store de_vhom_b


*Figure 2 / Table 5
*Predicted left-right positions by combinations of parental ideology


*1a. baseline estimation (full model)
regress lire0509 i.lire_m_dum1 i.lire_m_dum3 i.lire_v_dum1 i.lire_v_dum3 ib2.siops_father i.isced0509_3 ///
female age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36

outreg2 using RR2_GSOEP_LR_margins.xls, replace ctitle (M1) label

test 1.lire_m_dum1 = 1.lire_v_dum1 
test 1.lire_m_dum3 = 1.lire_v_dum3 

*1b. interact parental ideology only for certain categories
*manual interaction of only certain categories
regress lire0509 i.lire_m_dum1##i.lire_v_dum1 i.lire_m_dum3##i.lire_v_dum3 ib2.siops_father i.isced0509_3 ///
female age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
outreg2 using RR2_GSOEP_LR_margins.xls, append ctitle (M2 int) label

margins lire_m_dum1, over (lire_v_dum1) post

quietly regress lire0509 i.lire_m_dum1##i.lire_v_dum1 i.lire_m_dum3##i.lire_v_dum3 ib2.siops_father ///
i.isced0509_3 female age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36

margins lire_m_dum3, over (lire_v_dum3) post


*2a. three-way interaction with gender
regress lire0509 i.lire_m_dum1##i.lire_v_dum1##i.female i.lire_m_dum3##i.lire_v_dum3##i.female ///
ib2.siops_father i.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
outreg2 using RR2_GSOEP_LR_margins.xls, append ctitle (M3 int) label

margins lire_m_dum1, over(lire_v_dum1) post, if female==1
est store lire_int_left_de_g

quietly regress lire0509 i.lire_m_dum1##i.lire_v_dum1##i.female i.lire_m_dum3##i.lire_v_dum3##i.female ///
ib2.siops_father i.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
margins lire_m_dum1, over(lire_v_dum1) post , if female==0
est store lire_int_left_de_b

quietly regress lire0509 i.lire_m_dum1##i.lire_v_dum1##i.female i.lire_m_dum3##i.lire_v_dum3##i.female ///
ib2.siops_father i.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
margins lire_v_dum3, over (lire_m_dum3) post, if female==1
est store lire_int_right_de_g

quietly regress lire0509 i.lire_m_dum1##i.lire_v_dum1##i.female i.lire_m_dum3##i.lire_v_dum3##i.female ///
ib2.siops_father i.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
margins lire_v_dum3, over (lire_m_dum3) post, if female==0
est store lire_int_right_de_b

quietly regress lire0509 i.lire_m_dum1##i.lire_v_dum1##i.female i.lire_m_dum3##i.lire_v_dum3##i.female ///
ib2.siops_father i.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
margins r.female, over(i.lire_m_dum1 i.lire_v_dum1) post

quietly regress lire0509 i.lire_m_dum1##i.lire_v_dum1##i.female i.lire_m_dum3##i.lire_v_dum3##i.female ///
ib2.siops_father i.isced0509_3 age0509 i.year_lr ib3.region, cluster(hhnr), if age0509<36
margins r.female, over(i.lire_m_dum3 i.lire_v_dum3) post






************
***SHP***
************

*Swiss Household Panel, 1999-2017
*Data version used: W1-19; DOI: 10.23662/FORS-DS-932-2
*see https://www.da-ra.de/dara/study/web_show?res_id=667299&mdlang=en&detail=true


*DATASET WORKING FILE CREATION*


global datadest "Desktop/Data_file_long_creation_W1_19/datadest"
set more off

cd  "Desktop/SHP_waves_1_19"


*adding yearly variables for all the waves, long data file creation:

local year = 1999 // this is a macro, it has the value 1999
  
foreach y in 99 00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15 16 17 {
use idpers status?? idspou?? idhous?? plingu?? noga2m?? p??w31 is4maj?? xis4ma?? x??w03 ///
educat?? civsta?? age?? relarp?? occupa?? p??p10 p??p19 p??p01 tr1maj?? using shp`y'_p_user if status`y'==0, clear
renvars status?? idspou?? idhous?? noga2m?? is4maj?? xis4ma?? ///
educat?? civsta?? age?? relarp?? occupa?? tr1maj?? , postsub(`y') 
renvars p`y'* , presub(p`y' p)
renvars x`y'* , presub(x`y' x)
gen year=`year' // replaces `year' with the value of the macro 
save $datadest/tempp`y', replace
      local year = `year' + 1 // changes the value of the macro
   }
  

foreach y in 99 00 01 02 03 04 05 06 07 08 09 11 14 17 {
use idpers p??p13 p??p15 p??p17 using shp`y'_p_user, clear
renvars p`y'* , presub(p`y' p)
save $datadest/temp_prefs`y', replace
   } 
 
 
foreach y in 99 00 01 02 03 04 05 06 07 08 09 11 14 17 {
use $datadest/tempp`y'
merge 1:1 idpers using $datadest/temp_prefs`y', nogen
keep if status==0
save $datadest/tempp`y', replace
}

foreach y in 00 01 02 03 04 05 06 07 08 09 11 14 17 {
use idpers p??p20 p??p21 p??p22 using shp`y'_p_user, clear
renvars p`y'* , presub(p`y' p)
save $datadest/temp_gender`y', replace
   } 
   
foreach y in 00 01 02 03 04 05 06 07 08 09 11 14 17 {
use $datadest/tempp`y'
merge 1:1 idpers using $datadest/temp_gender`y', nogen
keep if status==0
save $datadest/tempp`y', replace
}

foreach y in 99 00 01 02 03 04 05 06 07 08 09 12 15 {
use idpers p??r01 p??r04 using shp`y'_p_user, clear
renvars p`y'* , presub(p`y' p)
save $datadest/temp_reli`y', replace
   }
   
   foreach y in 99 00 01 02 03 04 05 06 07 08 09 12 15 {
use $datadest/tempp`y'
merge 1:1 idpers using $datadest/temp_reli`y', nogen
keep if status==0
save $datadest/tempp`y', replace
}

*append the waves
use $datadest/tempp17
foreach y in 16 15 14 13 12 11 10 09 08 07 06 05 04 03 02 01 00 99 {
      append using $datadest/tempp`y'
   }
   
save $datadest/longfile_child, replace
xtset idpers year
save $datadest/longfile_child, replace

*add social origin data
use $datadest/longfile_child
merge m:1 idpers using shp_so, keepusing(p__o13 p__o14 p__p47 p__p46 p__o58 p__o59 p__o34 p__o17 is4faj__ tr1faj__) keep (master match)  nogen
save $datadest/longfile_child_so, replace

*add biography data 
use $datadest/longfile_child_so
merge m:1 idpers using shp0_bvfe_user, keepusing(bvfe003 bvfe004) keep (master match) nogen
save $datadest/longfile_child_so_bio, replace


*add master personal file variables
use $datadest/longfile_child_so_bio
merge m:1 idpers using shp_mp, keepusing(sex birthy idfath__ idmoth__) keep (master match) nogen
save $datadest/longfile_child_so_bio_mp, replace


*make parent files
use $datadest/longfile_child_so_bio_mp
renvars _all, postfix(_m)
drop idmoth___m // delete the mother's ID variable of the mothers
rename idpers_m idmoth__ // rename the person ID into mother's ID
rename year_m year
xtset idmoth__ year
save $datadest/longfile_mother, replace

use $datadest/longfile_child_so_bio_mp, clear
renvars _all, postfix(_v)
drop idfath___v // delete the father's ID variable
rename idpers_v idfath__ // rename the person ID into father's ID
rename year_v year
xtset idfath__ year
save $datadest/longfile_father, replace

*merge with child's file
use $datadest/longfile_child_so_bio_mp

bysort idpers: keep if _n==1
keep idpers
save $datadest/idperslist.dta, replace
clear
set obs 19
gen year= 1998 + _n
cross using $datadest/idperslist.dta
merge 1:1 idpers year using $datadest/longfile_child_so_bio_mp, nogen
recode _all(-1=.a)(-2=.b)(-3=.c)(-4=.d)(-5=.e)(-6=.f)(-7=.g)(-8=.h) 

egen idmoth__2=mode(idmoth__), by(idpers)
egen idfath__2=mode(idfath__), by(idpers)
drop idfath__ idmoth__
rename idmoth__2 idmoth__
rename idfath__2 idfath__

merge m:1 idmoth__ year using $datadest/longfile_mother, nogen keep (master match)
xtset idpers year 
save $datadest/longfile_child_mother, replace

merge m:1 idfath__ year using $datadest/longfile_father, nogen keep (master match)
save $datadest/SHP_workfile, replace

xtset idpers year
save $datadest/SHP_workfile, replace


*RECODING


use SHP_workfile.dta

recode _all(-1=.a)(-2=.b)(-3=.c)(-4=.d)(-5=.e)(-6=.f)(-7=.g)(-8=.h) 

*left/right
gen lire=pp10
gen lire_m=pp10_m
gen lire_v=pp10_v
xttab lire 

*lagged l/r 
gen lire_c=lire
replace lire_c=l.lire if lire_c>=.

*identify sample
sort idpers year
drop if lire>=.
bysort idpers: gen total= _N
bysort idpers: gen num_wave= _n
gen last_obs=1 if total==num_wave

tab year last_obs

*replacing the missing values for the parental ideology with lags
gen lire_mc = lire_m
replace lire_mc= L.lire_mc if lire_mc>=.
tab1 lire_m lire_mc

gen lire_vc = lire_v
replace lire_vc= L.lire_vc if lire_vc>=.
tab1 lire_v lire_vc

tab lire_mc if last_obs==1
tab lire lire_mc if last_obs==1

tab year if last_obs==1


*difference to father and mother ideology
gen liredif_v=lire-lire_vc if lire_c<. & lire_vc<.
tab liredif_v

gen liredif_m=lire-lire_mc if lire_c<. & lire_mc<.
tab liredif_m 

gen lireabdif_v=abs(liredif_v)
gen lireabdif_m=abs(liredif_m)
tab1 lireabdif_v lireabdif_m

recode lireabdif_m (0/1=1 "Distance 0-1")(2/3=2 "Distance 2-3")(4/max=3 "Distance 4 or more"), gen (lireabdifm_3)
recode lireabdif_v (0/1=1 "Distance 0-1")(2/3=2 "Distance 2-3")(4/max=3 "Distance 4 or more"), gen (lireabdifv_3)

// recode parental ideology in groups

recode lire_mc (0/2=1 "left") (3/4=2 "moderate left") (5=3 "centre") (6/7=4 "moderate right") (8/10=5 "right"), gen (lire_m_5)
recode lire_vc (0/2=1 "left") (3/4=2 "moderate left") (5=3 "centre") (6/7=4 "moderate right") (8/10=5 "right"), gen (lire_v_5)

*left-right in 3 blocks
capture drop lire_3 lirem_3 lirev_3
recode lire lire_mc lire_vc (0/4=1) (5=2) (6/10=3), gen(lire_3 lirem_3 lirev_3)
label define lire3 1 "left" 2 "centre" 3 "right"
label values lire_3 lirem_3 lirev_3 lire3
tab1 lire_3 lirem_3 lirev_3

*corresponding parental homogeneity measure
gen par_hom=.
replace par_hom=1 if (lirem_3==lirev_3) & lirem_3<. & lirev_3<.
replace par_hom=0 if (lirem_3!=lirev_3) & lirem_3<. & lirev_3<.
tab par_hom

*left/right-wing ideology
capture drop left
recode lire_3 (1=1) (2=0) (3=0), gen(left)
label define left 1 "left" 0 "centre/right"
label values left left
tab1 lire_3 left

capture drop right
recode lire_3 (1=0) (2=0) (3=1), gen(right)
label define right 1 "right" 0 "left/centre"
label values right right
tab1 lire_3 right

*centre category (for t-testing)
recode lire_3 (1=0) (2=1) (3=0), gen(centre)
label define centre 1 "centre" 0 "left/right"
label values centre centre
tab1 lire_3 centre

*left and right categories also for parents
capture drop left_m
recode lirem_3 (1=1) (2=0) (3=0), gen(left_m)
label values left_m left
tab1 lirem_3 left_m

recode lirem_3 (1=0) (2=0) (3=1), gen(right_m)
label values right_m right
tab1 lirem_3 right_m

recode lirev_3 (1=1) (2=0) (3=0), gen(left_v)
label values left_v left
tab1 lirev_3 left_v

recode lirev_3 (1=0) (2=0) (3=1), gen(right_v)
label values right_v right
tab1 lirev_3 right_v

*create new value labels for parental ideology
gen lirem_3_i=lirem_3
tab1 lirem_3 lirem_3_i
label define lire3m 1 "Mother Left" 2 "Mother Centre" 3 "Mother Right"
label values lirem_3_i lire3m
tab lirem_3_i

gen lirev_3_i=lirev_3
tab1 lirev_3 lirev_3_i
label define lire3v 1 "Father Left" 2 "Father Centre" 3 "Father Right"
label values lirev_3_i lire3v
tab lirev_3_i

*create dummies for parental ideology
tab lirem_3, gen(lire_m_dum)
tab lirev_3, gen(lire_v_dum)

*combinations of parental ideology (for descriptives)
gen par_combi=.
replace par_combi=1 if lirem_3==1 & lirev_3==1 // both left
replace par_combi=2 if lirem_3==1 & lirev_3==2 // mother left, father centre
replace par_combi=3 if lirem_3==1 & lirev_3==3 // mother left, father right

replace par_combi=4 if lirem_3==3 & lirev_3==1 // father left, mother right
replace par_combi=5 if lirem_3==2 & lirev_3==1 // father left, mother centre

replace par_combi=6 if lirem_3==3 & lirev_3==3 // both right
replace par_combi=7 if lirem_3==3 & lirev_3==2 // father centre, mother right
replace par_combi=8 if lirem_3==2 & lirev_3==3 // mother centre, father right
replace par_combi=0 if lirem_3==2 & lirev_3==2 // both centre 

label define par_combi 0 "both centre" 1 "both left" 2 "M left, F centre" 3 "M left, F right" ///
4 "F left, M right" 5 "F left, M centre" 6 "both right" 7 "F centre, M right" 8 "M centre, F right" 0 "both centre"

label values par_combi par_combi
tab par_combi

*similarity in ideological block parent-child
gen mc_hom=.
replace mc_hom=1 if (lirem_3==lire_3) & lirem_3<. & lire_3<.
replace mc_hom=0 if (lirem_3!=lire_3) & lirem_3<. & lire_3<.
tab mc_hom
tab lirem_3 lire_3

gen vc_hom=.
replace vc_hom=1 if (lirev_3==lire_3) & lirev_3<. & lire_3<.
replace vc_hom=0 if (lirev_3!=lire_3) & lirev_3<. & lire_3<.
tab vc_hom
tab lirev_3 lire_3

*gender
gen female=0
replace female=1 if sex==2

*level of education
recode educat (0=0 "still in school") (1/3=1 "low") (4/5=2 "medium") (6/10=3 "high"), gen(educat_3)
tab1 educat educat_3

*age
bysort idpers: gen age=year-birthy

*SIOPS
capture drop treiman_v
tab tr1faj__,m
gen treiman_v=tr1faj__ // this is retrospective var

// Add this with prospective data for children who entered the panel before age of 20 
gen treiman_v2=treiman_v
replace treiman_v2=tr1maj_v if age==15 & (tr1maj_v<. & treiman_v2>=.) //prospective but at age 15
replace treiman_v2=tr1maj_v if age==16 & (tr1maj_v<. & treiman_v2>=.)
summarize treiman_v2, detail
tab treiman_v2 // look at 30th percentile and 70th percentile for cut-off scores
recode treiman_v2 (min/36=1 "low") (37/46=2 "medium") (47/max=3 "high"), gen(vsiops_3)
tab vsiops_3 if last_obs==1

*replace with previous obs if missing
replace vsiops_3=L.vsiops_3 if vsiops_3>=.
tab vsiops_3 if last_obs==1 // 500 obs more

*replace with prospective info on father's job if still missing
recode tr1maj_v (min/36=1 "low") (37/46=2 "medium") (47/max=3 "high"), gen(tr1maj_v_3) 
replace vsiops_3=tr1maj_v_3 if (tr1maj_v_3<. & vsiops_3>=.)
tab vsiops_3 if last_obs==1


keep if last_obs==1

save SHP_workfile_recoded_1wavepp.dta, replace



*ANALYSIS

use SHP_workfile_recoded_1wavepp.dta


**Descriptive analysis with analytic sample

quietly regress lire ib2.lirem_3_i ib2.lirev_3_i female ib2.vsiops_3 ib1.educat_3 age i.year, cluster(idmoth__), if age>15 & age<36

*Distance in left-right ideology 
tab lireabdifm_3 if e(sample)
tab lireabdifv_3 if e(sample) 

tab lireabdifm_3, gen(liredifm_dum)
tab lireabdifv_3, gen(liredifv_dum)

ttest liredifm_dum1=liredifv_dum1 if e(sample)

summarize lireabdif_m if e(sample) 
summarize lireabdif_v if e(sample) 

*correlation
pwcorr lire lire_mc if e(sample)
pwcorr lire lire_vc if e(sample) 

*Table 1 and 2: Overlap in ideological blocks
tab lire_3 lirem_3, col, if e(sample)
tab lire_3 lirev_3, col, if e(sample)

*t-tests
ttest right=centre if lirev_3==3 & e(sample)
ttest right=centre if lirem_3==3 & e(sample)


* Table 3: Parental similarity in ideology
tab lirem_3 lirev_3, col, if e(sample)
*smallest category: n=34 (mother right, father left) and n=108 (mother right, father centre)

tab lire_m_5 lire_v_5, summarize(lire) means freq, if e(sample)

tab lire_3 par_combi, col, if par_hom==1 & e(sample)
tab lire_3 par_combi, col, if par_hom==0 & e(sample)

*gender gap in parent and offspring generations
ttest lire, by(female),  if e(sample)

ttest lire_mc == lire_vc, unpaired, if e(sample)

*descriptives mother
summarize lire if e(sample) 
summarize lire_mc if e(sample) 
tab1 left left_m if e(sample)
tab lire_3 if e(sample) 
tab lirem_3 if e(sample) 
ttest lire=lire_mc if e(sample)
ttest left=left_m if e(sample) 

*descriptives father
summarize lire if e(sample) 
summarize lire_vc if e(sample) 
tab1 left left_v if e(sample) 
tab lire_3 if e(sample) 
tab lirev_3 if e(sample)
ttest lire=lire_vc if e(sample)


ttest lire, by(female), if e(sample)
tab lire_3 if e(sample) 
tab lire_3 if female==1 & e(sample)
tab lire_3 if female==0 & e(sample)


*comparison of lagged accounts of parental ideology
tab1 lire_m lire_mc if e(sample) // 1808 vs 2025 (diff is 217)
tab lire_mc lire_m if e(sample), miss // 217
// 217 of 2025 = (217/20.25) 11% of mothers

tab1 lire_v lire_vc if e(sample) // 1848 vs 2025 (diff is 177)
tab lire_vc lire_v if e(sample), miss // 177 
// 177 of 2025 = (177/20.25) 9% of fathers

*postestimation
quietly regress vc_hom ib3.lirev_3##i.female par_hom ib2.vsiops_3 ib1.educat_3 age i.year, cluster(idmoth__), if age>15 & age<36

predict vchom_pred
tab vchom_pred 

quietly regress mc_hom ib3.lirem_3##i.female par_hom ib2.vsiops_3 ib1.educat_3 age i.year, cluster(idmoth__), if age>15 & age<36
predict mchom_pred
tab mchom_pred 



*Regressions


Figure 1 / Table 4: probability of being in the same block

//MOTHERS
*1. baseline model
regress mc_hom ib3.lirem_3 par_hom ib2.vsiops_3 ib1.educat_3 female age i.year, cluster(idmoth__), if age>15 & age<36
outreg2 using RR_SHP_LR_hom.xls, replace ctitle (Mothers 1)

*2. interaction with gender
regress mc_hom ib3.lirem_3##i.female par_hom ib2.vsiops_3 ib1.educat_3 age i.year, cluster(idmoth__), if age>15 & age<36
outreg2 using RR_SHP_LR_hom.xls, append ctitle (Mothers 2)


margins lirem_3, over(female) post

**compare significance of difference between boys and girl for LW mother
quietly regress mc_hom ib3.lirem_3##i.female par_hom ib2.vsiops_3 ib1.educat_3 age i.year, cluster(idmoth__), if age>15 & age<36
margins r.female, over(lirem_3) post

quietly regress mc_hom ib3.lirem_3##i.female par_hom ib2.vsiops_3 ib1.educat_3 age i.year, cluster(idmoth__), if age>15 & age<36
margins lirem_3, over(female) post, if female==1
est store ch_mhom_g

quietly regress mc_hom ib3.lirem_3##i.female par_hom ib2.vsiops_3 ib1.educat_3 age i.year, cluster(idmoth__), if age>15 & age<36
margins lirem_3, over(female) post, if female==0
est store ch_mhom_b



//FATHERS


*1. baseline model
regress vc_hom ib3.lirev_3 par_hom ib2.vsiops_3 ib1.educat_3 female age i.year, cluster(idmoth__), if age>15 & age<36
outreg2 using RR_SHP_LR_hom.xls, append ctitle (Fathers 1)

*2. interaction with gender
regress vc_hom ib3.lirev_3##i.female par_hom ib2.vsiops_3 ib1.educat_3 age i.year, cluster(idmoth__), if age>15 & age<36
outreg2 using RR_SHP_LR_hom.xls, append ctitle (Fathers 2)

margins r.female, over(lirev_3) post

quietly regress vc_hom ib3.lirev_3##i.female par_hom ib2.vsiops_3 ib1.educat_3 age i.year, cluster(idmoth__), if age>15 & age<36
margins lirev_3, over(female) post, if female==1
est store ch_vhom_g

quietly regress vc_hom ib3.lirev_3##i.female par_hom ib2.vsiops_3 ib1.educat_3 age i.year, cluster(idmoth__), if age>15 & age<36
margins lirev_3, over(female) post, if female==0
est store ch_vhom_b




*Figure 2 / Table 5
*Predicted left-right positions by combinations of parental ideology

*1a. baseline estimation (full model)
regress lire i.lire_m_dum1 i.lire_m_dum3 i.lire_v_dum1 i.lire_v_dum3 ib2.vsiops_3 ib1.educat_3 ///
female age i.year, cluster(idmoth__), if age>15 & age<36

outreg2 using RR2_SHP_LR_margins.xls, replace ctitle (M1) label

test 1.lire_m_dum1 = 1.lire_v_dum1 
test 1.lire_m_dum3 = 1.lire_v_dum3 


*1b. interact parental ideology only for certain categories
regress lire i.lire_m_dum1##i.lire_v_dum1 i.lire_m_dum3##i.lire_v_dum3 ib2.vsiops_3 ib1.educat_3 ///
female age i.year, cluster(idmoth__), if age>15 & age<36

outreg2 using RR2_SHP_LR_margins.xls, append ctitle (M2 int) label

margins lire_m_dum1, over (lire_v_dum1) post

quietly regress lire i.lire_m_dum1##i.lire_v_dum1 i.lire_m_dum3##i.lire_v_dum3 ib2.vsiops_3 ib1.educat_3 ///
female age i.year, cluster(idmoth__), if age>15 & age<36

margins lire_m_dum3, over (lire_v_dum3) post

*2a. three-way interaction with gender
regress lire i.lire_m_dum1##i.lire_v_dum1##i.female i.lire_m_dum3##i.lire_v_dum3##i.female ib2.vsiops_3 ///
ib1.educat_3 female age i.year, cluster(idmoth__), if age>15 & age<36

outreg2 using RR2_SHP_LR_margins.xls, append ctitle (M3 int) label

margins lire_m_dum1, over(lire_v_dum1) post, if female==1
est store lire_int_left_ch_g

quietly regress lire i.lire_m_dum1##i.lire_v_dum1##i.female i.lire_m_dum3##i.lire_v_dum3##i.female ///
ib2.vsiops_3 ib1.educat_3 female age i.year, cluster(idmoth__), if age>15 & age<36
margins lire_m_dum1, over(lire_v_dum1) post , if female==0
est store lire_int_left_ch_b

quietly regress lire i.lire_m_dum1##i.lire_v_dum1##i.female i.lire_m_dum3##i.lire_v_dum3##i.female ///
ib2.vsiops_3 ib1.educat_3 female age i.year, cluster(idmoth__), if age>15 & age<36
margins lire_v_dum3, over (lire_m_dum3) post, if female==1
est store lire_int_right_ch_g

quietly regress lire i.lire_m_dum1##i.lire_v_dum1##i.female i.lire_m_dum3##i.lire_v_dum3##i.female ///
ib2.vsiops_3 ib1.educat_3 female age i.year, cluster(idmoth__), if age>15 & age<36
margins lire_v_dum3, over (lire_m_dum3) post, if female==0
est store lire_int_right_ch_b

quietly regress lire i.lire_m_dum1##i.lire_v_dum1##i.female i.lire_m_dum3##i.lire_v_dum3##i.female ///
ib2.vsiops_3 ib1.educat_3 female age i.year, cluster(idmoth__), if age>15 & age<36
margins r.female, over(i.lire_m_dum1 i.lire_v_dum1) post

quietly regress lire i.lire_m_dum1##i.lire_v_dum1##i.female i.lire_m_dum3##i.lire_v_dum3##i.female ///
ib2.vsiops_3 ib1.educat_3 female age i.year, cluster(idmoth__), if age>15 & age<36
margins r.female, over(i.lire_m_dum3 i.lire_v_dum3) post






*FIGURES

set scheme cleanplots

*Figure 1 DE & CH
coefplot (de_mhom_g, label(Daughters)) (de_mhom_b, label(Sons)), bylabel("GERMANY" "Mothers-Offspring") ///
|| (de_vhom_g) (de_vhom_b), bylabel("GERMANY" "Fathers-Offspring") ///
|| (ch_mhom_g, label(Daughters)) (ch_mhom_b, label(Sons)), bylabel("SWITZERLAND" "Mothers-Offspring") ///
|| (ch_vhom_g) (ch_vhom_b), bylabel("SWITZERLAND" "Fathers-Offspring") ///
||, xscale(lstyle(none)) yscale(lstyle(none)) byopts(legend(position(3))) ///
coeflabels (1.lire0509m_3 = "Parent Left" 2.lire0509m_3 = "Parent Centre" 3.lire0509m_3 = "Parent Right" ///
, notick labgap(2)) ///
rename (1.lire0509v_3 = 1.lire0509m_3 2.lire0509v_3 = 2.lire0509m_3 3.lire0509v_3 = 3.lire0509m_3 ///
1.lirev_3 = 1.lire0509m_3 2.lirev_3 = 2.lire0509m_3 3.lirev_3 = 3.lire0509m_3 ///
1.lirem_3 = 1.lire0509m_3 2.lirem_3 = 2.lire0509m_3 3.lirem_3 = 3.lire0509m_3) graphregion(margin(l=0)) plotregion(lcolor(gs10) lwidth(small))
graph save "Figure1.gph"



*Figure 2 DE & CH
coefplot (lire_int_left_de_g, label(Daughters) msymbol(oh) mcolor(red) ciopts(lcolor(red))) (lire_int_right_de_g, label(Daughters) msymbol(oh) mcolor(red) ciopts(lcolor(red))) (lire_int_left_de_b, label(Sons) msymbol(o) mcolor(ltblue) ciopts(lcolor(ltblue))) (lire_int_right_de_b, label(Sons) msymbol(o) mcolor(ltblue) ciopts(lcolor(ltblue))), bylabel(Germany) ///
|| (lire_int_left_ch_g, label(Daughters) msymbol(oh) mcolor(red) ciopts(lcolor(red))) (lire_int_right_ch_g, label(Daughters) msymbol(oh) mcolor(red) ciopts(lcolor(red))) (lire_int_left_ch_b, label(Sons) msymbol(o) mcolor(ltblue) ciopts(lcolor(ltblue))) (lire_int_right_ch_b, label(Sons) msymbol(o) mcolor(ltblue) ciopts(lcolor(ltblue))), bylabel(Switzerland) ///
||, drop(0.lire_v_dum1#0.lire_m_dum1 0.lire_m_dum3#0.lire_v_dum3) ///
coeflabels (1.lire_v_dum1#1.lire_m_dum1 = "{bf:Mother Left * Father Left}" ///
0.lire_v_dum1#1.lire_m_dum1 = "Mother Left * Father not Left" ///
1.lire_v_dum1#0.lire_m_dum1 = "Father Left * Mother not Left" ///
1.lire_m_dum3#0.lire_v_dum3 = "Mother Right * Father not Right" ///
0.lire_m_dum3#1.lire_v_dum3 = "Father Right * Mother not Right" ///
1.lire_m_dum3#1.lire_v_dum3 = "{bf:Mother Right * Father Right}", notick labgap(2)) ///
order(1.lire_v_dum1#1.lire_m_dum1) ///
xscale (range (2 7)) xlabel(2 (1) 7) xscale(lstyle(none)) ///
yscale(lstyle(none)) xline(5) plotregion(lcolor(gs10)) legend(rows(2))
graph save "Figure2.gph"



