####################################################################
# PREPARE DATA for replication from original data files file for:  #
# Lehrer et al. "Rallying Around the Leader in Times of Crises:    #
# "The Opposing Effects of Perceived Threat and Anxiety"           #
#                                                                  #
# Published in the European Journal of Political Research          #
####################################################################

########################################################################
# Last edited:                                                         #
# 26 July 2024 by Roni Lehrer                                          #
#                                                                      #
# Information on package versions can be found at the end of this file #
########################################################################

## clear working memory
rm(list=ls())

## load packages
library(data.table)
library(haven)

## Read MCS data
dat <- as.data.table(read_dta("ZA7745_v1-0-0.dta"))
dat <- dat[Teilnahme==1] #keep only row of respondents that provide data in a given MCS week

## Adjust date format
dates <- strsplit(dat$dDatum, ".", TRUE)
dat$date <- as.Date(sapply(dates, function(x) paste0(x[3], "-", x[2], "-", x[1])))
rm(dates)


#####################
# VARIABLE RECODING #
#####################

#Merkel appoval (MCS)
dat$merkel1 <- ifelse(dat$SCPX001<0, NA, dat$SCPX001-1)

#Altmaier appoval
dat$altmaier <- ifelse(dat$SCPX003<0, NA, dat$SCPX003-1)

#Spahn skalometer
dat$spahn <- ifelse(dat$SCPX002<0, NA, dat$SCPX002-1)

#perceived threat
dat$threat <- ifelse(dat$SCBX003<0, NA, dat$SCBX003)/10

#anxiety index
dat$worried <- ifelse(dat$SCBX009<0, NA, dat$SCBX009)
dat$nervous <- ifelse(dat$SCBX011<0, NA, dat$SCBX011)
dat$fear <- (dat$worried+dat$nervous -2)/6

#gender
dat$male <- ifelse(dat$gender_19<0, NA,
              ifelse(dat$gender_19==1, 1, 0))

#age
dat$age.cat <- ifelse(dat$year_of_birth_cat_19<=0, NA,
               ifelse(dat$year_of_birth_cat_19<=6, 1,0))
dat$age.cat <- factor(dat$age.cat, levels=0:1, labels=c("-60", "60+"))

##income per capita in household in previous month
dat$hh.incom.prev.month <- ifelse(dat$Woche%in%c(1,2), dat$SCDX001, #March
                           ifelse(dat$Woche%in%c(3,4,5,6), dat$SCDX005, #April
                           ifelse(dat$Woche%in%c(7,8,9,10,11), dat$SCDX007, #May
                           ifelse(dat$Woche%in%c(12,13,14,15), dat$SCDX008, dat$SCDX009)))) #June, July
dat$hh.incom.prev.month <-  ifelse(dat$hh.incom.prev.month<0, NA,
                            ifelse(dat$hh.incom.prev.month==1, 150,
                            ifelse(dat$hh.incom.prev.month==2, 400,
                            ifelse(dat$hh.incom.prev.month>2&dat$hh.incom.prev.month<15,
                                (dat$hh.incom.prev.month*500)-500, 7500))))
dat$number_hh_members_19 <- ifelse(dat$number_hh_members_19<0, NA, dat$number_hh_members_19)
dat$hh.incom.prev.month.pc <-  dat$hh.incom.prev.month/dat$number_hh_members_19
dat$hh.incom.prev.month.pc <- dat$hh.incom.prev.month.pc/1000

#medical precondition (was asked only once and needs to be assigned to all observation for a respondent)
dat$precondition <- ifelse(dat$SCTX001<0, NA,
                    ifelse(dat$SCTX001==1, 1, 0))

source("functions/panel_expand.R") #load function panel.expand

dat <- dat[,precondition:=panel.expand(precondition), id_g]

#Are government policy and respondent preference for open/closed borders identical?
dat$demand.borders <- ifelse(dat$SCPX006_b<0, NA, dat$SCPX006_b)
dat$meets.borders <- ifelse(is.na(dat$demand.borders), NA,
                            ifelse((dat$demand.borders==1&dat$date<"2020-06-15")|(dat$demand.borders==0&dat$date>="2020-06-15"),
                                   1, 0))

###############################################
# Obtain information from pre-COVID GIP waves #
###############################################
#Merkel satisfaction from GIP Wave 36 (Juli 2018) (ZA6956)
gip <- as.data.table(read_dta("ZA6956_v1-0-0.dta"))
gip <- gip[,list(id_g, CE36233)]
dat <- merge(dat, gip, by="id_g", all.x=TRUE)
dat$merkel0 <- ifelse(dat$CE36233<0, NA, dat$CE36233-1)
dat$merkel0 <- dat$merkel0/10
rm(gip)

#Government satisfaction from GIP Wave 44 (November 2019) (ZA7593)
gip <- as.data.table(read_dta("ZA7593_v1-1-0.dta"))
gip <- gip[,list(id_g, CE44152)]
dat <- merge(dat, gip, by="id_g", all.x=TRUE)
dat$sat.breg.nov19 <- ifelse(dat$CE44152<0, NA, dat$CE44152)
dat$sat.breg.nov19 <- (dat$sat.breg.nov19-1)/10
rm(gip)

#################################################
## Merge with data of COVID-19 cases in Germany #
#################################################
load("data_for_preparation/rki.daily.pop.Rdata")
rki.daily <- unique(rki.daily[!is.na(last7days.cases.bund),list(date, last7days.cases.bund)])
dat <- merge(dat, rki.daily, by="date")
rm(rki.daily)

prepared.data <- dat[,list(date, id_g, Woche, wi_rake, merkel0, merkel1,
                         altmaier, spahn, threat, fear, age.cat, male,
                         hh.incom.prev.month.pc, precondition, meets.borders, sat.breg.nov19,
                         last7days.cases.bund)]

save(prepared.data, file="prepared_data.Rdata")




#####################################################################
# Information on R version, packages, and operating system settings #
#####################################################################

## R version 4.4.1 (2024-06-14 ucrt)
## Platform: x86_64-w64-mingw32/x64
## Running under: Windows 10 x64 (build 19045)

## Matrix products: default


## locale:
## [1] LC_COLLATE=German_Germany.utf8  LC_CTYPE=German_Germany.utf8
## [3] LC_MONETARY=German_Germany.utf8 LC_NUMERIC=C
## [5] LC_TIME=German_Germany.utf8

## time zone: Europe/Berlin
## tzcode source: internal

## attached base packages:
## [1] stats     graphics  grDevices utils     datasets  methods   base

## other attached packages:
## [1] haven_2.5.4       data.table_1.15.4

## loaded via a namespace (and not attached):
##  [1] utf8_1.2.4      R6_2.5.1        tzdb_0.4.0      magrittr_2.0.3
##  [5] glue_1.7.0      tibble_3.2.1    pkgconfig_2.0.3 lifecycle_1.0.4
##  [9] readr_2.1.5     cli_3.6.3       fansi_1.0.6     vctrs_0.6.5
## [13] compiler_4.4.1  forcats_1.0.0   tools_4.4.1     hms_1.1.3
## [17] pillar_1.9.0    rlang_1.1.4
